import React, { useState } from "react";
import { connect, useDispatch } from "react-redux";
import { useTranslation } from "react-i18next";
import { getApiState } from "../../selectors";
import { createAdminExport, setCreateAdminExportModalVisibility } from "./exportActions";
import { getAdminExportModalVisibility } from "./selectors";
import { values as _values } from "lodash";
import { Button, Checkbox, Modal, RadioButton, TextInput } from "@lib/components/lego";
import LabModalHeader from "../../utils/LabModalHeader";

import "./style.scss";

const LABELS = {
    keywordId: "keyword_id *",
    segmentId: "segment_id",
};

const KEYWORD_ID = {
    544: "544",
    545: "545",
    546: "546",
    547: "547",
    548: "548",
    549: "549",
    557: "557",
    601: "601",
    602: "602",
};

const EXPORT_TYPE = {
    heuristic: {
        value: "HEURISTIC",
        keywords: [KEYWORD_ID["547"], KEYWORD_ID["548"], KEYWORD_ID["549"], KEYWORD_ID["557"]],
    },
    probabilistic: {
        value: "PROBABILISTIC",
        keywords: [KEYWORD_ID["544"], KEYWORD_ID["545"], KEYWORD_ID["546"], KEYWORD_ID["557"]],
    },
    shortterm: {
        value: "SHORTTERM",
        keywords: [KEYWORD_ID["602"]],
    },
    longterm: {
        value: "LONGTERM",
        keywords: [KEYWORD_ID["601"]],
    },
};

const DEFAULT_EXPORT_TYPES = {
    544: EXPORT_TYPE.probabilistic.value,
    545: EXPORT_TYPE.probabilistic.value,
    546: EXPORT_TYPE.probabilistic.value,
    547: EXPORT_TYPE.heuristic.value,
    548: EXPORT_TYPE.heuristic.value,
    549: EXPORT_TYPE.heuristic.value,
    557: EXPORT_TYPE.heuristic.value,
    601: EXPORT_TYPE.longterm.value,
    602: EXPORT_TYPE.shortterm.value,
};

const CreateAdminSegmentExportModal = connect((state) => ({
    api: getApiState(state),
    visible: getAdminExportModalVisibility(state),
}))((props) => {
    const { api, visible, segmentId } = props;

    const [exportType, setExportType] = useState(EXPORT_TYPE.heuristic.value);
    const [exportKeywordId, setExportKeywordId] = useState(null);
    const [exportSegmentId, setExportSegmentId] = useState("");
    const [cryptaIdAsExportId, setCryptaIdAsExportId] = useState(false);

    const dispatch = useDispatch();
    const { t, i18n } = useTranslation("segments");

    const createExport_ = (eType, eKeywordId, eSegmentId) => {
        dispatch(createAdminExport(api, i18n.language, segmentId, eType, eKeywordId, eSegmentId));
    };

    const createExportCryptaId = (eType, eKeywordId, eSegmentId) => {
        dispatch(createAdminExport(api, i18n.language, segmentId, eType, eKeywordId, eSegmentId, true));
    };

    const handleSelectExportType = (value) => {
        setExportType(value);
    };

    const handleSelectKeywordId = (value) => {
        setExportKeywordId(value);
        setExportType(DEFAULT_EXPORT_TYPES[value]);
    };

    const handleSegmentIdChanged = (text) => {
        setExportSegmentId(text);
    };

    const handleCreateExport = () => {
        const cryptaIdExport = exportKeywordId === "557" && cryptaIdAsExportId;

        if (cryptaIdExport) {
            createExportCryptaId(exportType, exportKeywordId, exportSegmentId);
        } else {
            createExport_(exportType, exportKeywordId, exportSegmentId);
        }

        dispatch(setCreateAdminExportModalVisibility(false));
    };

    const isInputValid = () => {
        return (
            exportType &&
            exportKeywordId &&
            _values(EXPORT_TYPE)
                .filter((eType) => eType.value === exportType)[0]
                .keywords.includes(exportKeywordId)
        );
    };

    return (
        <Modal theme="normal" visible={visible} onClose={() => dispatch(setCreateAdminExportModalVisibility(false))}>
            <LabModalHeader
                title={t("createForm.export.creation")}
                hasClose
                onClose={() => dispatch(setCreateAdminExportModalVisibility(false))}
            />
            <div className="AddExportModal-content">
                <table className="AddExportModal-content-table">
                    <tbody>
                        <tr>
                            <td className="AddExportModal-label">{LABELS.keywordId}</td>
                            <td>
                                <RadioButton
                                    value={exportKeywordId}
                                    view="default"
                                    size="s"
                                    onChange={(event) => {
                                        handleSelectKeywordId(event.target.value);
                                    }}
                                    options={_values(KEYWORD_ID).map((keywordId) => ({
                                        value: keywordId,
                                        children: keywordId,
                                    }))}
                                />
                            </td>
                        </tr>

                        <tr>
                            <td className="AddExportModal-label">{t("createForm.export.type")}</td>
                            <td>
                                <RadioButton
                                    value={exportType}
                                    view="default"
                                    size="s"
                                    onChange={(event) => {
                                        handleSelectExportType(event.target.value);
                                    }}
                                    options={_values(EXPORT_TYPE).map((eType) => ({
                                        value: eType.value,
                                        children: eType.value,
                                        disabled: !eType.keywords.includes(exportKeywordId),
                                    }))}
                                />
                            </td>
                        </tr>

                        <tr>
                            <td className="AddExportModal-label">{LABELS.segmentId}</td>
                            <td>
                                <TextInput
                                    view="default"
                                    size="m"
                                    hasClear
                                    value={exportSegmentId}
                                    placeholder={LABELS.segmentId}
                                    onChange={(event) => handleSegmentIdChanged(event.target.value)}
                                />
                            </td>
                        </tr>

                        {exportKeywordId === "557" && (
                            <tr>
                                <td className="AddExportModal-crypta-id-checkbox">
                                    <Checkbox
                                        size="m"
                                        view="default"
                                        onChange={() => setCryptaIdAsExportId(!cryptaIdAsExportId)}
                                        checked={cryptaIdAsExportId}
                                    />
                                </td>
                                <td
                                    className="AddExportModal-crypta-id-label"
                                    onClick={() => setCryptaIdAsExportId(!cryptaIdAsExportId)}
                                >
                                    {t("createForm.export.cryptaId")}
                                </td>
                            </tr>
                        )}
                    </tbody>
                </table>

                <div className="SegmentExportCreateModal-button">
                    <Button view="action" size="s" disabled={!isInputValid()} onClick={handleCreateExport}>
                        {t("createForm.export.create")}
                    </Button>
                </div>
            </div>
        </Modal>
    );
});

export default CreateAdminSegmentExportModal;
