import React, { useEffect, useState } from "react";
import { connect } from "react-redux";
import { filter as _filter, map as _map, trim as _trim } from "lodash";
import ConditionCardToolsBar from "../rule/ConditionCardToolsBar";
import ConditionValues from "../rule/ConditionValues";
import { getActiveIdsState, getEditableExportState } from "../selectors";
import { deleteExpressions, fetchSegmentExport, storeExpressions } from "../actions";
import { getAccessState, getActiveSegmentInfoState, getApiState } from "../../selectors";
import { allowedToEditSegment } from "../../utils/roles";

import "../rule/style.scss";
import "./ExpressionsConditionCard.scss";

const ExpressionsConditionCard = connect((state) => ({
    api: getApiState(state),
    isAdmin: getAccessState(state).Admin,
    notAllowedSegments: getAccessState(state).RestrictedSegments,
    segmentInfo: getActiveSegmentInfoState(state).value,
    exportId: getActiveIdsState(state).export,
    storedFullExpressions: getEditableExportState(state).value.fullExpressions,
}))((props) => {
    const { api, isAdmin, notAllowedSegments, exportId, segmentInfo, storedFullExpressions, dispatch } = props;

    const [rawValues, setRawValues] = useState([]);
    const [isEditable, setIsEditable] = useState(false);
    const [displayFormat, setFormat] = useState("text");

    const getRawValues = () => {
        return storedFullExpressions ? storedFullExpressions.map((each) => each.raw) : [];
    };

    const initialize = () => {
        setRawValues(getRawValues());
    };

    useEffect(() => {
        dispatch(fetchSegmentExport(api, exportId));
    }, []);

    useEffect(() => {
        initialize();
    }, [storedFullExpressions]);

    const handleEditClick = () => {
        setIsEditable(true);
    };

    const handleDisplayFormatChange = (newFormat) => {
        setFormat(newFormat);
    };

    const handleAcceptEditing = () => {
        let values = _filter(
            _map(rawValues, (each) => _trim(each)),
            (each) => each
        );

        // TODO(cherenkov-p-a) Make storeExpressions update expressions
        dispatch(storeExpressions(api, exportId, rawValues));

        setIsEditable(false);
        setRawValues(values);
    };

    const handleRejectEditing = (rawValues) => {
        setIsEditable(false);
        setRawValues(rawValues);
    };

    const handleDeleteCondition = () => {
        const message = `[${exportId}] Do you want to delete expressions?`;
        if (window.confirm(message)) {
            dispatch(dispatch(deleteExpressions(api, exportId)));
        }
    };

    const handleAddValue = (value) => {
        setRawValues([...rawValues, value]);
    };

    const handleChangeValue = (index, value) => {
        let newValues = [...rawValues];
        newValues[index] = value;

        setRawValues(newValues);
    };

    const handleRemoveValue = (index) => {
        let newValues = _filter(rawValues, (each, i) => i !== index);
        setRawValues(newValues);
    };

    const handleUpdateAllValues = (values) => {
        setRawValues(values);
    };

    const canEditExpressions = () => {
        return isAdmin || (allowedToEditSegment(segmentInfo) && !notAllowedSegments);
    };

    return (
        <div className="ExpressionsConditionCard-container rule-condition-card">
            <ConditionCardToolsBar
                editing={isEditable}
                canEdit={canEditExpressions()}
                format={displayFormat}
                onEditClick={handleEditClick}
                onFormatChange={handleDisplayFormatChange}
                onAcceptEditing={handleAcceptEditing}
                onRejectEditing={() => handleRejectEditing(getRawValues())}
                onDeleteCondition={handleDeleteCondition}
            />
            <ConditionValues
                source={"EXPRESSIONS"}
                editing={isEditable}
                format={displayFormat}
                rawValues={rawValues}
                fullValues={storedFullExpressions ? storedFullExpressions : []}
                onAddValue={handleAddValue}
                onChangeValue={handleChangeValue}
                onRemoveValue={handleRemoveValue}
                onUpdateAllValues={handleUpdateAllValues}
            />
        </div>
    );
});

export default ExpressionsConditionCard;
