import React, { useState } from "react";
import { connect } from "react-redux";
import { useTranslation } from "react-i18next";
import { isEmpty } from "lodash";
import { Button, Checkbox, RadioButton, TextInput } from "@lib/components/lego";
import { getCurrentUserLogin } from "../../utils/cookie";
import { getSearchPanelState } from "./filterSelectors";
import SearchInput from "../tree/SearchInput";
import DisplayModeSelector from "./DisplayModeSelector";
import CreateSegmentButton from "../tree/CreateSegmentButton";
import EditTreeButton from "../tree/EditTreeButton";
import { getApiState } from "../../selectors";
import Tag from "./Tag";

import "./style.scss";
import { matchSegmentConditions } from "../actions";
import TagsInput from "./TagsInput";

const TagsOperator = {
    OR: "OR",
    AND: "AND",
};

const SearchOptions = connect((state) => ({
    api: getApiState(state),
    searchPanelVisible: getSearchPanelState(state).visible,
}))((props) => {
    const { api, searchPanelVisible, dispatch } = props;

    const user = getCurrentUserLogin();

    const [searchParams, setSearchParams] = useState({
        author: "",
        stakeholder: "",
        tags: new Set(),
        name: "",
        exportSegmentId: "",
        exportKeywordId: "",
        ticket: "",
    });

    const [withoutAuthor, setWithoutAuthor] = useState(false);
    const [withoutStakeholder, setWithoutStakeholder] = useState(false);
    const [newTagValue, setNewTagValue] = useState("");
    const [tagsOperator, setTagsOperator] = useState(TagsOperator.OR);
    const [showDeleted, setShowDeleted] = useState(false);

    const { t } = useTranslation("segments");

    // Fields description: https://a.yandex-team.ru/arc_vcs/crypta/lab/proto/segment_conditions.proto
    const conditions = {
        author: searchParams.author,
        emptyAuthor: withoutAuthor,
        stakeholder: searchParams.stakeholder,
        emptyStakeholder: withoutStakeholder,
        tags: Array.from(searchParams.tags),
        tagsCondition: tagsOperator,
        ticket: searchParams.ticket,
        showDeleted: showDeleted,
    };

    if (!isEmpty(searchParams.exportKeywordId)) {
        conditions.exportKeywordId = parseInt(searchParams.exportKeywordId);
    }

    if (!isEmpty(searchParams.exportSegmentId)) {
        conditions.exportSegmentId = parseInt(searchParams.exportSegmentId);
    }

    const search = () => {
        dispatch(matchSegmentConditions(api, conditions));
    };

    const changeSearchParams = (values) => {
        setSearchParams({
            ...searchParams,
            ...values,
        });
    };

    const setSelfAsStakeholder = () => {
        setSearchParams({
            ...searchParams,
            stakeholder: user,
        });
    };

    const setSelfAsAuthor = () => {
        setSearchParams({
            ...searchParams,
            author: user,
        });
    };

    const enterNewTagItem = (value) => {
        setNewTagValue(value);
    };

    const addTag = () => {
        if (!isEmpty(newTagValue)) {
            let tags = searchParams.tags;
            tags.add(newTagValue);

            setSearchParams({
                ...searchParams,
                tags: tags,
            });

            setNewTagValue("");
        }
    };

    const removeTag = (value) => {
        let tags = searchParams.tags;
        tags.delete(value);

        setSearchParams({
            ...searchParams,
            tags: tags,
        });
    };

    const extendedSearchOptions = (
        <div className="SearchOptions-extended">
            <div className="SearchOptions-author-and-stakeholder">
                <span className="SearchOptions-label-author">{t("search.author")}</span>
                <TextInput
                    className="SearchOptions-input"
                    view="default"
                    value={searchParams.author}
                    size="s"
                    hasClear
                    baseline
                    onChange={(event) => changeSearchParams({ author: event.target.value })}
                    placeholder={t("search.loginPlaceholder")}
                    disabled={withoutAuthor}
                />

                <Button view="default" size="s" onClick={setSelfAsAuthor} className="SearchOptions-me">
                    {t("search.me")}
                </Button>

                <Checkbox
                    className="SearchOptions-label"
                    label={t("search.withoutAuthor")}
                    size="s"
                    view="default"
                    onChange={() => setWithoutAuthor(!withoutAuthor)}
                    checked={withoutAuthor}
                />

                <span className="SearchOptions-label">{t("search.stakeholder")}</span>
                <TextInput
                    className="SearchOptions-input"
                    view="default"
                    value={searchParams.stakeholder}
                    size="s"
                    hasClear
                    baseline
                    onChange={(event) => changeSearchParams({ stakeholder: event.target.value })}
                    placeholder={t("search.loginPlaceholder")}
                    disabled={withoutStakeholder}
                />

                <Button view="default" size="s" onClick={setSelfAsStakeholder} className="SearchOptions-me">
                    {t("search.me")}
                </Button>

                <Checkbox
                    className="SearchOptions-label"
                    label={t("search.withoutStakeholder")}
                    size="s"
                    view="default"
                    onChange={() => setWithoutStakeholder(!withoutStakeholder)}
                    checked={withoutStakeholder}
                />
            </div>
            <div className="SearchOptions-tags-and-ids-options">
                <span className="SearchOptions-label-tags">{t("search.tags")}</span>
                <div className="SearchOptions-tags">
                    <div className="SearchOptions-tags-input-group">
                        <TagsInput
                            currentValue={newTagValue}
                            onChange={(event) => enterNewTagItem(event.target.value)}
                            addTag={addTag}
                        />
                        <RadioButton
                            value={tagsOperator}
                            view="default"
                            size="s"
                            onChange={(event) => setTagsOperator(event.target.value)}
                            options={[
                                { value: TagsOperator.OR, children: TagsOperator.OR },
                                { value: TagsOperator.AND, children: TagsOperator.AND, disabled: true },
                            ]}
                            baseline
                        />
                    </div>

                    <div className="SearchOptions-tags-list">
                        {Array.from(searchParams.tags).map((tag) => {
                            return (
                                <Tag key={tag} onClose={() => removeTag(tag)}>
                                    {tag}
                                </Tag>
                            );
                        })}
                    </div>
                </div>

                <div className="SearchOptions-export-segment-id">
                    <span className="SearchOptions-label">Segment ID</span>
                    <TextInput
                        view="default"
                        size="s"
                        value={searchParams.exportSegmentId}
                        hasClear
                        baseline
                        onChange={(event) => changeSearchParams({ exportSegmentId: event.target.value })}
                        placeholder="ID"
                    />
                </div>

                <div className="SearchOptions-keyword-id">
                    <span className="SearchOptions-label">Keyword ID</span>
                    <TextInput
                        view="default"
                        size="s"
                        value={searchParams.exportKeywordId}
                        hasClear
                        baseline
                        onChange={(event) => changeSearchParams({ exportKeywordId: event.target.value })}
                        placeholder="ID"
                    />
                </div>
            </div>

            <div className="SearchOptions-ticket">
                <span className="SearchOptions-label-ticket">Ticket</span>
                <TextInput
                    className="SearchOptions-input"
                    view="default"
                    size="s"
                    value={searchParams.ticket}
                    hasClear
                    baseline
                    onChange={(event) => changeSearchParams({ ticket: event.target.value })}
                    placeholder="TASK-42"
                />
                <Checkbox
                    className="SearchOptions-label"
                    label={t("search.showDeleted")}
                    size="s"
                    view="default"
                    onChange={() => setShowDeleted(!showDeleted)}
                    checked={showDeleted}
                />
            </div>

            <div className="SearchOptions-actions">
                <Button view="action" size="s" onClick={search}>
                    {t("search.apply")}
                </Button>
            </div>
        </div>
    );

    return (
        <div className="SearchOptions">
            <div className="SearchOptions-basic">
                <div className="SearchOptions-basic-left">
                    <SearchInput />
                    <DisplayModeSelector />
                </div>

                <span className="SearchOptions-basic-right">
                    <CreateSegmentButton />
                    <EditTreeButton />
                </span>
            </div>

            {searchPanelVisible && extendedSearchOptions}
        </div>
    );
});

export default SearchOptions;
