import React, { useEffect } from "react";
import { connect } from "react-redux";
import { push } from "connected-react-router";
import qs from "query-string";
import { useTranslation } from "react-i18next";
import { isUndefined } from "lodash";
import Loading from "@lib/components/loading";

import { getActiveIdsState, getActiveRuleIdState, getApiState, getLoadedState } from "../selectors";
import { fetchSegmentsAndGroups } from "./actions";

import Failable from "../utils/Failable";
import SegmentsMenu from "./menu/index";
import SegmentCard from "./card/index";
import StatsModal, { RuleEstimateStatsModal, RuleConditionEstimateStatsModal } from "./StatsModal";
import RuleModal from "./RuleModal";
import ExpressionsModal from "./expression/ExpressionsModal";
import LalModal from "./lal/LalModal";
import DisplayedSegments from "./DisplayedSegments";

import "./style.scss";

const Segments = connect((state) => ({
    api: getApiState(state),
    loaded: getLoadedState(state),
    failed: state.base.failed,
    message: state.base.message,
    groupId: getActiveIdsState(state).group,
    segmentId: getActiveIdsState(state).segment,
    exportId: getActiveIdsState(state).export,
    ruleId: getActiveRuleIdState(state),
}))((props) => {
    const { api, groupId, segmentId, exportId, ruleId, message, loaded, failed, dispatch } = props;

    const { i18n } = useTranslation();
    let lang = i18n.language;

    const initializeActiveGroup = (activeGroup, activeSegment) => {
        if (isUndefined(activeGroup) && isUndefined(activeSegment)) {
            dispatch(
                push({
                    search:
                        "?" +
                        qs.stringify({
                            group: "root",
                            segment: "root",
                        }),
                })
            );
        }
    };

    useEffect(() => {
        initializeActiveGroup(groupId, segmentId);
    }, []);

    useEffect(() => {
        dispatch(fetchSegmentsAndGroups(api, lang));
    }, [lang]);

    return (
        <Failable failed={failed} errorMessage={message}>
            <Loading loaded={loaded}>
                <div className="SegmentsLayoutBody">
                    <SegmentsMenu />
                    <DisplayedSegments />
                    <SegmentCard />
                </div>
                <StatsModal key={"segment-stats" + segmentId + exportId + lang} />
                <RuleEstimateStatsModal key={"rule-estimate-stats" + segmentId + exportId + lang} />
                <RuleConditionEstimateStatsModal key={"rule-condition-estimate-stats" + segmentId + exportId + lang} />
                <RuleModal key={"rule" + ruleId + lang} />
                <ExpressionsModal key={"expression" + segmentId + exportId + lang} />
                <LalModal key={"lal" + segmentId + exportId + lang} />
            </Loading>
        </Failable>
    );
});

export default Segments;
