import React, { useEffect, useState } from "react";
import { connect } from "react-redux";
import { useTranslation } from "react-i18next";
import { Select, TextInput, Slider, Checkbox } from "@lib/components/lego";
import { deleteLal, fetchSegmentExport, storeLal } from "../actions";
import SegmentFormPropertyTooltip from "../../utils/tooltips";
import FinishChangesButtons from "../utils/FinishChangesButtons";

import "./LalCard.scss";
import { allowedToEditSegment } from "../../utils/roles";
import {
    getAccessState,
    getActiveIdsState,
    getActiveSegmentInfoState,
    getEditableExportState,
    getApiState,
} from "../../selectors";
import ActionButton from "../utils/ActionButton";
import { isEmpty } from "lodash";

const LalCard = connect((state) => ({
    api: getApiState(state),
    isAdmin: getAccessState(state).Admin,
    exportId: getActiveIdsState(state).export,
    exportInfo: getEditableExportState(state).value,
    segment: getActiveSegmentInfoState(state).value,
    storedLal: getEditableExportState(state).value.lal,
}))((props) => {
    const { api, isAdmin, exportId, segment, storedLal, dispatch, exportInfo } = props;
    const [currentLal, setCurrentLal] = useState({});
    const [previousLal, setPreviousLal] = useState({});
    const [isEditable, setIsEditable] = useState(false);
    const [maxCoverageVisible, setMaxCoverageVisible] = useState(false);
    const [sizeSliderVisible, setSizeSliderVisible] = useState(true);
    const [checked, setChecked] = useState(false);

    const { t } = useTranslation(["segments", "common"]);

    const getRuleIds = () => {
        if (!isEmpty(exportInfo.ruleId)) {
            return [exportInfo.ruleId];
        } else {
            let ruleIds = [];
            for (const exp of segment.exports.exports) {
                if (!isEmpty(exp.ruleId)) {
                    ruleIds.push(exp.ruleId);
                }
            }
            return ruleIds;
        }
    };

    const updateVisibility = (checked) => {
        setMaxCoverageVisible(checked);
        setSizeSliderVisible(!checked);
        if (!checked) {
            updateMaxCoverage(0);
        }
    };

    const initialLalState = {
        includeInput: "",
        exponent: 0,
        maxCoverage: 0,
        ruleId: getRuleIds()[0],
    };

    const initialize = () => {
        setCurrentLal(storedLal && !isEmpty(storedLal) ? JSON.parse(storedLal) : initialLalState);
        if (storedLal && !isEmpty(storedLal)) {
            let maxCoverage = JSON.parse(storedLal).maxCoverage;

            if (maxCoverage && maxCoverage !== 0) {
                setChecked(true);
            } else {
                setChecked(false);
            }
        }
    };

    useEffect(() => {
        dispatch(fetchSegmentExport(api, exportId));
    }, []);

    useEffect(() => {
        initialize();
    }, [storedLal]);

    useEffect(() => {
        updateVisibility(checked);
    }, [checked]);

    const updateRuleId = (value) => {
        setCurrentLal({
            ...currentLal,
            ruleId: value,
        });
    };

    const updateIncludeInput = (value) => {
        setCurrentLal({
            ...currentLal,
            includeInput: value,
        });
    };

    const updateMaxCoverage = (value) => {
        setCurrentLal({
            ...currentLal,
            maxCoverage: value,
            exponent: 0,
        });
    };

    const updateExponent = (value) => {
        setCurrentLal({
            ...currentLal,
            exponent: value[0],
        });
    };

    const handleEditClick = () => {
        setPreviousLal(currentLal);
        setIsEditable(true);
    };

    const handleAcceptEditing = () => {
        dispatch(storeLal(api, exportId, JSON.stringify(currentLal)));
        setPreviousLal(currentLal);
        setIsEditable(false);
    };

    const handleRejectEditing = () => {
        setIsEditable(false);
        setCurrentLal(previousLal);
        if (previousLal.maxCoverage !== 0) {
            setChecked(true);
        } else {
            setChecked(false);
        }
    };

    const handleDeleteLal = () => {
        const message = `[${exportId}] Do you want to delete lal?`;
        if (window.confirm(message)) {
            setChecked(false);
            setIsEditable(false);
            dispatch(dispatch(deleteLal(api, exportId)));
        }
    };

    const canEditLal = () => {
        return isAdmin || allowedToEditSegment(segment);
    };

    const renderRightPanel = () => {
        return (
            <span>
                {!isEditable && (
                    <ActionButton
                        action="edit"
                        view="clear"
                        theme="raised"
                        title={t("common:actions.edit")}
                        size="s"
                        onClick={handleEditClick}
                    />
                )}
                <ActionButton
                    action="delete"
                    view="clear"
                    theme="raised"
                    title={t("common:actions.delete")}
                    size="s"
                    onClick={handleDeleteLal}
                />
            </span>
        );
    };

    const isValidInput = () => {
        return (typeof currentLal.includeInput == "boolean");
    };

    return (
        <div className="LalCard-container">
            <div className="lal-edit-tools-bar">
                <span className="lal-edit-tools-bar-right">
                    {isEditable && (
                        <FinishChangesButtons
                            acceptDisabled={!isValidInput()}
                            onAccept={handleAcceptEditing}
                            onReject={handleRejectEditing}
                        />
                    )}
                    {canEditLal() && renderRightPanel()}
                </span>
            </div>
            <table className="LalParameters">
                <tbody>
                    <tr>
                        <td className="LalLabel">{t("segments:lal.labels.ruleId")}</td>
                        <td></td>
                        <td className="LalValue">
                            <Select
                                size="m"
                                view="default"
                                width="max"
                                onChange={(event) => updateRuleId(event.target.value)}
                                value={currentLal.ruleId}
                                options={getRuleIds().map((ruleId) => ({ value: ruleId, content: ruleId }))}
                                disabled={!isEditable}
                            />
                        </td>
                        <td className="LalLabel">
                            <SegmentFormPropertyTooltip text={t("segments:lal.tooltips.rule")} />
                        </td>
                    </tr>
                    <tr>
                        <td className="LalLabel">{t("segments:lal.labels.includeInputLabel")}</td>
                        <td></td>
                        <td className="LalValue">
                            <Select
                                size="m"
                                view="default"
                                width="max"
                                onChange={(event) => updateIncludeInput(event.target.value)}
                                value={currentLal.includeInput}
                                options={[
                                    { value: true, content: t("segments:lal.labels.includeInput") },
                                    { value: false, content: t("segments:lal.labels.excludeInput") },
                                ]}
                                disabled={!isEditable}
                            />
                        </td>
                        <td className="LalLabel">
                            <SegmentFormPropertyTooltip text={t("segments:lal.tooltips.core")} />
                        </td>
                    </tr>
                    {sizeSliderVisible && (
                        <tr>
                            <td className="LalLabel">{t("segments:lal.labels.size.small")}</td>
                            <td></td>
                            <td className="LalValue">
                                <Slider
                                    filled
                                    step={1}
                                    min={0}
                                    max={4}
                                    view="default"
                                    value={[currentLal.exponent]}
                                    onChange={(event, value) => updateExponent(value)}
                                    onInput={(event, value) => updateExponent(value)}
                                    disabled={!isEditable}
                                />
                            </td>
                            <td className="LalLabel">{t("segments:lal.labels.size.big")}</td>
                        </tr>
                    )}
                    <tr>
                        <td className="LalLabel">{t("segments:lal.labels.maxCoverageLabel")}</td>
                        <td className="LalCheckBox">
                            <Checkbox
                                label=""
                                size="s"
                                view="default"
                                onChange={() => setChecked(!checked)}
                                checked={checked}
                                disabled={!isEditable}
                            />
                        </td>
                        {maxCoverageVisible && (
                            <td>
                                <TextInput
                                    className="MaxCoverageValue"
                                    view="default"
                                    theme="normal"
                                    size="s"
                                    type="number"
                                    value={currentLal.maxCoverage}
                                    onChange={(event) => updateMaxCoverage(parseInt(event.target.value))}
                                    disabled={!isEditable}
                                />
                            </td>
                        )}
                        {maxCoverageVisible && (
                            <td className="LalLabel">
                                <SegmentFormPropertyTooltip text={t("segments:lal.tooltips.maxCoverage")} />
                            </td>
                        )}
                    </tr>
                </tbody>
            </table>
        </div>
    );
});

export default LalCard;
