import React, { useEffect, useState } from "react";
import { connect } from "react-redux";
import { isEmpty, replace as _replace, trim as _trim } from "lodash";
import { useTranslation } from "react-i18next";
import FontAwesome from "react-fontawesome";
import { Modal, Button } from "@lib/components/lego";

import Date from "../../utils/Date";
import ConditionCardToolsBar from "./ConditionCardToolsBar";
import ConditionValues from "./ConditionValues";
import {
    getAccessState,
    getActiveConditionState,
    getActiveRuleIdState,
    getActiveSegmentAttributesState,
    getApiState,
} from "../../selectors";
import {
    approveRuleCondition,
    deleteRuleCondition,
    putRuleCondition,
    resetRequestConditionApproval,
    setRequestApprovalVisibility,
} from "../actions";
import HistoryTooltip from "./HistoryTooltip";
import InfoTooltip from "./InfoTooltip";
import PrecalculatedTableCondition from "./precalculated-table";
import ConditionCardCoverageBar from "./ConditionCardCoverageBar";


const ConditionCard = connect((state) => ({
    segment: getActiveSegmentAttributesState(state),
    api: getApiState(state),
    activeCondition: getActiveConditionState(state),
    ruleId: getActiveRuleIdState(state),
    isAdmin: getAccessState(state).Admin,
}))((props) => {
    const { isAdmin, activeCondition, ruleId, dispatch, canEdit, api } = props;

    const [rawValues, setRawValues] = useState([]);
    const [format, setFormat] = useState("text");
    const [isEditingMode, setEditingMode] = useState(false);
    const [visibleError, setVisibleError] = useState(false);
    const [tooLong, setTooLong] = useState(false);

    const { t, i18n } = useTranslation(["segments", "condition"]);
    const lang = i18n.language;

    const getRawValues = () => {
        return activeCondition && activeCondition.fullValues ? activeCondition.fullValues.map((each) => each.raw) : [];
    };

    const initialize = () => {
        setRawValues(getRawValues());
    };

    useEffect(() => {
        initialize();
    }, [activeCondition]);

    const handleFormatChange = (format) => {
        setFormat(format);
    };

    const handleEditClick = () => {
        setEditingMode(true);
        dispatch(setRequestApprovalVisibility(false));
        dispatch(resetRequestConditionApproval());
    };

    const handleUpdateAllValues = (newValues) => {
        setRawValues(newValues);
    };

    const handleAddNewValue = () => {
        let newValues = [...rawValues, ""];
        setRawValues(newValues);
    };

    const handleChangeValue = (index, value) => {
        let updatedValues = [...rawValues];
        updatedValues[index] = _replace(value, new RegExp("\\n", "g"), "");

        setRawValues(updatedValues);
    };

    const handleRemoveValue = (index) => {
        let updatedValues = [...rawValues];
        updatedValues.splice(index, 1);
        setRawValues(updatedValues);
    };

    const handleAcceptEditing = () => {
        let source = activeCondition.source;
        let trimmedValues = rawValues.map((value) => _trim(value));

        if (trimmedValues.length >= 2000) {
            setTooLong(true);
            setVisibleError(true);
        }

        dispatch(putRuleCondition(api, lang, ruleId, source, trimmedValues, setVisibleError));

        setEditingMode(false);
        dispatch(setRequestApprovalVisibility(true));
    };

    const handleRejectEditing = (rawValues) => {
        setEditingMode(false);
        setRawValues(rawValues);
        dispatch(setRequestApprovalVisibility(false));
    };

    const handleDeleteCondition = () => {
        const message = `[${ruleId}] Do you want to delete ${activeCondition.source}:${activeCondition.state} condition ?`;
        if (window.confirm(message)) {
            dispatch(deleteRuleCondition(api, lang, ruleId, activeCondition.source, activeCondition.state)).then(() =>
                props.onSourceChange(undefined, undefined)
            );
        }
    };

    const handleApproveCondition = () => {
        const message = `[${ruleId}] Do you want to approve ${activeCondition.source} condition ?`;
        if (window.confirm(message)) {
            dispatch(approveRuleCondition(api, lang, ruleId, activeCondition.source)).then(() =>
                props.onSourceChange(activeCondition.source, "APPROVED")
            );
        }
    };

    const needApprove = activeCondition ? activeCondition.state === "NEED_APPROVE" : false;
    const needUserApprove = isAdmin && needApprove;
    const conditionProblemStats = getConditionProblemStats(activeCondition);

    return !isEmpty(activeCondition) ? (
        <div className="rule-condition-card">
            <div className="rule-condition-card-header">
                <div className="rule-condition-source">
                    {!isEmpty(activeCondition.source) && t("condition:sourcesTitles." + activeCondition.source)}
                    <ConditionSourceDescriptionButton source={activeCondition.source ? activeCondition.source : ""} />
                    <ConditionHistoryButton condition={activeCondition} />
                    {conditionProblemStats.errors > 0 && (
                        <span className="error-container" title={t("segments:rule.format.errors")}>
                            <span>{conditionProblemStats.errors}</span>
                            <FontAwesome className="error" name="exclamation-triangle" />
                        </span>
                    )}
                    {conditionProblemStats.warnings > 0 && (
                        <span className="warning-container" title={t("segments:rule.format.warnings")}>
                            <span>{conditionProblemStats.warnings}</span>
                            <FontAwesome className="warning" name="exclamation-circle" />
                        </span>
                    )}
                </div>
                <div className="rule-condition-revision">
                    {"# "}
                    {activeCondition.revision}
                </div>
            </div>
            <ConditionCardCoverageBar/>
            <ConditionCardToolsBar
                editing={isEditingMode}
                canEdit={canEdit}
                format={format}
                needUserApprove={needUserApprove}
                onApprove={handleApproveCondition}
                onEditClick={handleEditClick}
                onFormatChange={handleFormatChange}
                onAcceptEditing={handleAcceptEditing}
                onRejectEditing={() => handleRejectEditing(getRawValues())}
                onDeleteCondition={handleDeleteCondition}
            />

            {activeCondition.source === "PRECALCULATED_TABLES" ? (
                <PrecalculatedTableCondition
                    values={activeCondition.values}
                    editingMode={isEditingMode}
                    updateConditionValues={handleUpdateAllValues}
                />
            ) : (
                <ConditionValues
                    source={activeCondition.source}
                    editing={isEditingMode}
                    format={format}
                    rawValues={rawValues}
                    fullValues={activeCondition.fullValues}
                    onAddValue={handleAddNewValue}
                    onChangeValue={handleChangeValue}
                    onRemoveValue={handleRemoveValue}
                    onUpdateAllValues={handleUpdateAllValues}
                />
            )}
            <Modal theme="normal" visible={visibleError} onClose={() => setVisibleError(false)}>
                <div className="rule-condition-error-popup">
                    <div className="rule-condition-error-popup-block">
                        <span className="rule-condition-error-popup-icon">
                            <FontAwesome className="error fa-2x" name="exclamation-triangle" />
                        </span>
                        {tooLong && (
                            <span className="rule-condition-error-popup-text">{t("segments:rule.tooLong")}</span>
                        )}
                        {!tooLong && (
                            <span className="rule-condition-error-popup-text">{t("segments:rule.hasErrors")}</span>
                        )}
                    </div>
                    <div className="rule-condition-error-popup-button">
                        <Button view="default" size="m" onClick={() => setVisibleError(false)}>
                            {t("segments:rule.ok")}
                        </Button>
                    </div>
                </div>
            </Modal>
        </div>
    ) : null;
});

const ConditionSourceDescriptionButton = (props) => {
    const { source } = props;

    const { t } = useTranslation("condition");

    const [isButtonChecked, checkButton] = useState(false);

    const handleButtonClick = () => {
        checkButton(!isButtonChecked);
    };

    return (
        <span className="ConditionCard-info-button-container">
            <Button
                key="condition-source-description-button"
                title="Info"
                view="clear"
                theme="clear"
                size="s"
                checked={isButtonChecked}
                onClick={handleButtonClick}
            >
                <FontAwesome name="info" />
            </Button>

            <InfoTooltip visible={isButtonChecked}>
                <div className="condition-source-description-popup-body">{t("condition:sourcesInfo." + source)}</div>

                <div className="InfoTooltip-close-button">
                    <Button view="pseudo" size="m" onClick={handleButtonClick}>
                        Close
                    </Button>
                </div>
            </InfoTooltip>
        </span>
    );
};

const ConditionHistoryButton = (props) => {
    const { condition } = props;

    const [isButtonChecked, checkButton] = useState(false);

    const handleButtonClick = () => {
        checkButton(!isButtonChecked);
    };

    return (
        <span className="ConditionCard-history-button-container">
            <Button
                title="History"
                view="clear"
                theme="clear"
                size="s"
                checked={isButtonChecked}
                onClick={handleButtonClick}
            >
                <FontAwesome name="history" />
            </Button>

            <HistoryTooltip visible={isButtonChecked}>
                <div className="condition-history-popup-body">
                    <div className="ConditionCard-history-row" title="Created">
                        <span className="ConditionCard-history-date-type-glyph">Created:</span>
                        <span className="ConditionCard-history-date">
                            <Date value={condition.timestamps.created} lang="en" />
                        </span>
                    </div>

                    <div className="ConditionCard-history-row" title="Modified">
                        <span className="ConditionCard-history-date-type-glyph">Modified:</span>
                        <span className="ConditionCard-history-date">
                            <Date value={condition.timestamps.modified} lang="en" />
                        </span>
                    </div>
                </div>

                <div className="HistoryTooltip-close-button">
                    <Button view="pseudo" size="m" onClick={handleButtonClick}>
                        Close
                    </Button>
                </div>
            </HistoryTooltip>
        </span>
    );
};

function getConditionProblemStats(condition) {
    let result = { errors: 0, warnings: 0 };
    if (condition == null || condition.fullValues == null) {
        return result;
    }

    for (let item of condition.fullValues) {
        if (item.error) {
            result.errors += 1;
        }
        if (item.tags.length > 0) {
            result.warnings += 1;
        }
    }

    return result;
}

export default ConditionCard;
