import React, { useEffect, useState } from "react";
import { useTranslation } from "react-i18next";

import {
    fetchRuleRevisionEstimateCoverage,
    fetchRuleRevisionEstimateStats,
    showRuleRevisionEstimateStatsModal,
} from "../actions";
import { connect } from "react-redux";
import { getActiveConditionState, getRuleRevisionEstimateCoverageState } from "../selectors";
import { getApiState } from "../../selectors";
import { isEmpty, isNil } from "lodash";
import Coverage from "./Coverage";
import UpdateEstimateStatsButton from "./UpdateEstimateStatsButton";

const CONDITIONS_WITH_ESTIMATES = new Set([
    "WORDS",
    "PUBLIC_WORDS",
    "BROWSER_TITLES",
    "METRICA_TITLES",
    "SEARCH_REQUESTS",
    "SITES",
    "PUBLIC_SITES",
    "BROWSER_SITES",
    "METRICA_SITES",
    "YANDEX_REFERRER",
    "APPS",
    "PRECALCULATED_TABLES",
    "SEARCH_RESULTS",
]);

const SPECIAL_COVERAGES = new Map([
    ["APPS", "rule.estimate.monthlyCoverage"],
    ["PRECALCULATED_TABLES", "rule.estimate.coverage"],
]);

const ConditionCardCoverageBar = connect((state) => ({
    activeCondition: getActiveConditionState(state),
    api: getApiState(state),
    value: getRuleRevisionEstimateCoverageState(state).value,
    loading: getRuleRevisionEstimateCoverageState(state).loading,
    failed: getRuleRevisionEstimateCoverageState(state).error,
}))((props) => {
    const { activeCondition, api, dispatch, failed, loading, value } = props;

    const { t, i18n } = useTranslation(["segments"]);
    const lang = i18n.language;

    const initialize = () => {
        if (hasEstimateStats(activeCondition) && !isNil(activeCondition.revision)) {
            dispatch(fetchRuleRevisionEstimateCoverage(api, activeCondition.revision, lang));
        }
    };

    useEffect(() => {
        initialize();
    }, [activeCondition]);

    const [tooltipVisible, setTooltipVisibility] = useState(false);

    const conditionHasEstimateStats = hasEstimateStats(activeCondition);

    const handleShowAndRequestStats = () => {
        if (!isEmpty(activeCondition) && !isNil(activeCondition.revision)) {
            dispatch(fetchRuleRevisionEstimateStats(api, activeCondition.revision, lang));
            dispatch(showRuleRevisionEstimateStatsModal(activeCondition));
        }
    };

    return (
        <span className="rule-condition-coverage-bar">
            <span>
                {t(SPECIAL_COVERAGES.get(activeCondition.source) || "rule.estimate.dailyCoverage")}
                <UpdateEstimateStatsButton />:{" "}
            </span>
            <Coverage
                loading={loading}
                failed={failed}
                value={value}
                lang={lang}
                hasEstimateStats={conditionHasEstimateStats}
                handleShowAndRequestStats={handleShowAndRequestStats}
                setTooltipVisibility={setTooltipVisibility}
                t={t}
            />
        </span>
    );
});

function hasEstimateStats(condition) {
    return !isEmpty(condition) && !isNil(condition.source) && CONDITIONS_WITH_ESTIMATES.has(condition.source);
}

export default ConditionCardCoverageBar;
