import React from "react";
import { connect, useDispatch } from "react-redux";

import { Button, Textarea, TooltipStateful } from "@lib/components/lego";
import FontAwesome from "react-fontawesome";
import {
    join as _join,
    split as _split,
    map as _map,
    replace as _replace,
    filter as _filter,
    repeat as _repeat,
    indexOf as _indexOf,
} from "lodash";
import { getActiveConditionState, getApiState } from "../../selectors";
import { fetchExportInfo, hideExportTooltip, showExportTooltip } from "../actions";
import { getExportTooltipVisibilityState } from "../selectors";
import ExportTooltip from "./ExportTooltip";
import { useTranslation } from "react-i18next";

const INDENT = "  ";
const ADD_INDENT_AFTER = ["("];
const REMOVE_INDENT_BEFORE = [")"];
const NO_NEW_LINE_AFTER = ["AND", "OR", "NOT"];
const WORD_RULES = new Set(["WORDS", "PUBLIC_WORDS", "SEARCH_REQUESTS", "BROWSER_TITLES"]);

const ConditionValues = connect((state) => ({
    api: getApiState(state),
    tooltipVisible: getExportTooltipVisibilityState(state).visible,
    tooltipIndex: getExportTooltipVisibilityState(state).activeIndex,
    activeCondition: getActiveConditionState(state),
}))((props) => {
    const { api, tooltipVisible, tooltipIndex, format, source, rawValues, fullValues, editing } = props;

    const getRawValuesFromString = (value) => {
        return _split(value, new RegExp("[\\n]", "g"));
    };

    const dispatch = useDispatch();

    const showExportTooltip_ = (index) => dispatch(showExportTooltip(index));
    const hideExportTooltip_ = () => dispatch(hideExportTooltip());
    const fetchExportTooltip = (api, exportId) => dispatch(fetchExportInfo(api, exportId));

    const handleMouseOverExport = (exportId, index) => {
        fetchExportTooltip(api, exportId);
        showExportTooltip_(index);
    };

    const renderValues = () => {
        if (format === "blocks") {
            return (
                <ConditionValuesBlocks
                    source={source}
                    rawValues={rawValues}
                    fullValues={fullValues}
                    onAddValue={props.onAddValue}
                    onChangeValue={props.onChangeValue}
                    onRemoveValue={props.onRemoveValue}
                    editing={editing}
                />
            );
        }
        if (format === "text") {
            return (
                <ConditionValuesText
                    source={source}
                    fullValues={fullValues}
                    onChange={(value) => props.onUpdateAllValues(getRawValuesFromString(value))}
                    editing={editing}
                    showTooltip={(exportId, index) => handleMouseOverExport(exportId, index)}
                    hideTooltip={hideExportTooltip_}
                    activeTooltipIndex={tooltipIndex}
                />
            );
        }
    };

    return <div className="ConditionValues">{renderValues()}</div>;
});

const ConditionValuesText = (props) => {
    const { fullValues, editing, showTooltip, hideTooltip, activeTooltipIndex, source } = props;

    const { t } = useTranslation("segments");

    const getValuesString = () => {
        return _join(
            fullValues.map((each) => {
                return each.raw;
            }),
            "\n"
        );
    };

    return (
        <div className="ConditionValues-text">
            {editing ? (
                <Textarea
                    view="default"
                    size="s"
                    value={getValuesString()}
                    onChange={(event) => props.onChange(event.target.value)}
                />
            ) : (
                <span className="ConditionValues-text-pretty">
                    <div>
                        <table className="rule-condition-table">
                            <tbody>
                                {fullValues.map((fullValue, lineIndex) => {
                                    return (
                                        <tr key={lineIndex}>
                                            <td className="rule-condition-row-value">
                                                {source === "EXPRESSIONS" ? (
                                                    highlightExports(
                                                        fullValue,
                                                        activeTooltipIndex,
                                                        showTooltip,
                                                        hideTooltip,
                                                        lineIndex
                                                    )
                                                ) : (
                                                    <span className="rule-condition-token">
                                                        {WORD_RULES.has(source)
                                                            ? replaceAndsWithSpaces(fullValue.normalized)
                                                            : fullValue.normalized}
                                                    </span>
                                                )}
                                            </td>
                                            <td className="rule-condition-status pull-right">
                                                <ConditionStatus fullValue={fullValue} t={t} />
                                            </td>
                                        </tr>
                                    );
                                })}
                            </tbody>
                        </table>
                    </div>
                </span>
            )}
        </div>
    );
};

const ConditionValuesBlocks = (props) => {
    const { rawValues, fullValues, source, editing } = props;

    const { t } = useTranslation("segments");

    if (editing) {
        return (
            <div className="ConditionValues-blocks">
                {rawValues &&
                    rawValues.map((each, index) => {
                        return (
                            <div key={index}>
                                <span className="rule-condition-block-editing">
                                    <Textarea
                                        view="default"
                                        size="s"
                                        value={each}
                                        hasClear
                                        onChange={(event) => props.onChangeValue(index, event.target.value)}
                                    />
                                    <Button
                                        view="clear"
                                        theme="clear"
                                        size="s"
                                        title="Remove value"
                                        onClick={() => props.onRemoveValue(index)}
                                    >
                                        <FontAwesome name="remove" />
                                    </Button>
                                </span>
                            </div>
                        );
                    })}
                <Button
                    view="clear"
                    theme="clear"
                    size="s"
                    title={t("expressions.add")}
                    onClick={(event) => props.onAddValue(event.target.value)}
                >
                    <FontAwesome name="plus" />
                </Button>
            </div>
        );
    } else {
        return (
            <div className="ConditionValues-blocks">
                <table className="rule-condition-table">
                    {fullValues &&
                        fullValues.map((each, index) => {
                            return (
                                <tr key={index}>
                                    <td className="rule-condition-row-value">
                                        <span className="rule-condition-block">
                                            <div className="rule-condition-block-pretty">
                                                {beautifyValue(source, each.normalized)}
                                            </div>
                                        </span>
                                    </td>
                                    <td className="rule-condition-status pull-right">
                                        <span className="rule-condition-block">
                                            <ConditionStatus fullValue={each} t={t} />
                                        </span>
                                    </td>
                                </tr>
                            );
                        })}
                </table>
            </div>
        );
    }
};

function beautifyValue(source, value) {
    return WORD_RULES.has(source) || source === "EXPRESSIONS" ? beautifyWords(value) : value;
}

function beautifyWords(text) {
    function getStringItems(text) {
        return _split(
            _replace(_replace(text, new RegExp("[(]", "g"), " ( "), new RegExp("[)]", "g"), " ) "),
            new RegExp("[\\s]", "g")
        );
    }

    let number_of_indents = 0;
    let new_line = true;

    return _join(
        _map(_filter(getStringItems(text)), (each) => {
            if (_indexOf(REMOVE_INDENT_BEFORE, each) !== -1) {
                number_of_indents -= 1;
            }
            let indent = _repeat(INDENT, number_of_indents);
            if (_indexOf(ADD_INDENT_AFTER, each) !== -1) {
                number_of_indents += 1;
            }

            let value = "";
            if (new_line) {
                value += indent;
            }
            value += each;
            if (_indexOf(NO_NEW_LINE_AFTER, each) === -1) {
                value += "\n";
                new_line = true;
            } else {
                value += " ";
                new_line = false;
            }
            return value;
        }),
        ""
    ).trim();
}

function highlightExports(fullValue, activeTooltipIndex, showTooltip, hideTooltip, lineIndex) {
    const EXPORT_PATTERN = /(export-[a-z0-9]{8})/i;

    let tokens = fullValue.normalized.split(EXPORT_PATTERN);

    return (
        <span>
            {tokens.map((token, index) => {
                let tooltipId = `${lineIndex}-${index}`;
                if (token.match(EXPORT_PATTERN) !== null) {
                    return (
                        <span
                            className="rule-condition-export"
                            key={index}
                            onMouseEnter={() => showTooltip(token, tooltipId)}
                            onMouseLeave={() => hideTooltip()}
                        >
                            {token}
                            {tooltipId === activeTooltipIndex && <ExportTooltip />}
                        </span>
                    );
                } else {
                    return (
                        <span className="rule-condition-token" key={index}>
                            {token}{" "}
                        </span>
                    );
                }
            })}
        </span>
    );
}

const ConditionStatus = (props) => {
    const { fullValue, t } = props;

    const fill = () => {
        if (fullValue.error) {
            return (
                <div className="error-container" title={t("rule.format.errors")}>
                    <FontAwesome className="error" name="exclamation-triangle" />
                </div>
            );
        } else if (fullValue.tags.length > 0) {
            return (
                <div className="warning-container" title={t("rule.format.warnings")}>
                    <FontAwesome className="warning" name="exclamation-circle" />
                </div>
            );
        } else {
            return <div />;
        }
    };

    return (
        <TooltipStateful size="s" view="default" content={lineFormatInfo(fullValue, t)} hasTail>
            <div className="rule-condition-status-container">{fill()}</div>
        </TooltipStateful>
    );
};

function replaceAndsWithSpaces(value) {
    value = _replace(value, new RegExp(" AND (?!NOT|\\()", "g"), " ");
    return _replace(value, new RegExp("\\) (?!AND|OR)", "g"), ") AND ");
}

function lineFormatInfo(fullValue, t) {
    if (fullValue == null) {
        return <div></div>;
    }

    return (
        <div className="rule-condition-tooltip">
            {fullValue.raw && (
                <div className="rule-condition-raw-value">{t("rule.format.raw") + ": " + fullValue.raw}</div>
            )}
            {fullValue.error && (
                <div className="rule-condition-error-message">{t("rule.format.error") + ": " + fullValue.error}</div>
            )}
            {fullValue.tags &&
                fullValue.tags.map((tag, tagIndex) => {
                    return <div className="rule-condition-format-tag">{t("rule." + tag)}</div>;
                })}
        </div>
    );
}

export default ConditionValues;
