import React, { useEffect, useState } from "react";
import { find as _find, isEmpty, isUndefined, some as _some, sortBy as _sortBy } from "lodash";
import { connect } from "react-redux";
import { useTranslation } from "react-i18next";

import ConditionsSelector from "./ConditionsSelector";
import ConditionCard from "./ConditionCard";
import { Button } from "@lib/components/lego";
import Link from "@lib/components/link";
import {
    getActiveSegmentAttributesState,
    getConditionApprovalState,
    getRequestApprovalVisibilityState,
    getRuleState,
} from "../selectors";
import { getCurrentUserLogin } from "../../utils/cookie";
import { requestConditionApproval, setActiveCondition } from "../actions";
import { getApiState } from "../../selectors";
import { setSource } from "./reducers";

import "./style.scss";

const Conditions = connect((state) => ({
    api: getApiState(state),
    rule: getRuleState(state),
    segment: getActiveSegmentAttributesState(state),
    requestApprovalVisible: getRequestApprovalVisibilityState(state),
    conditionApprovalResult: getConditionApprovalState(state).value,
    conditionApprovalInProgress: getConditionApprovalState(state).loading,
    conditionApprovalError: getConditionApprovalState(state).error,
}))((props) => {
    const {
        rule,
        requestApprovalVisible,
        conditionApprovalResult,
        conditionApprovalInProgress,
        conditionApprovalError,
        dispatch,
        canEdit,
    } = props;

    const [conditionSource, setConditionSource] = useState();
    const [conditionState, setConditionState] = useState();

    const { t } = useTranslation("segments");

    const initializeConditionsState = () => {
        let condition = getSortedConditions()[0];
        dispatch(setActiveCondition(condition));

        setConditionSource(!isEmpty(condition) ? condition.source : undefined);
        dispatch(setSource(condition ? condition.source : null));
    };

    useEffect(() => {
        initializeConditionsState();
    }, []);

    useEffect(() => {
        initializeConditionsState();
    }, [rule.id]);

    const handleConditionChange = (source, state) => {
        if (isUndefined(source)) {
            dispatch(setSource(null));
            initializeConditionsState();
        } else {
            setConditionSource(source);
            setConditionState(state);
            dispatch(setSource(source));
        }
    };

    const getSortedConditions = () => {
        return _sortBy(rule.conditions || [], ["source", "state"]);
    };

    const getCondition = (source, state) => {
        return _find(rule.conditions, (each) => each.source === source && each.state === state);
    };

    const handleRequestApproval = () => {
        const { api, segment } = props;

        const linkedIssueKey = isEmpty(segment.tickets) ? null : segment.tickets[0];
        const author = getCurrentUserLogin();

        dispatch(requestConditionApproval(api, rule.id, author, linkedIssueKey, location.href));
    };

    let sortedConditions = getSortedConditions();

    const needTicket = _some(rule.conditions, (condition) => condition.state === "NEED_APPROVE");

    return (
        rule.conditions && (
            <div>
                <ConditionsSelector
                    ruleId={rule.id}
                    conditions={sortedConditions}
                    activeSource={conditionSource}
                    activeState={conditionState}
                    onChange={handleConditionChange}
                    canEdit={canEdit}
                />
                <ConditionCard
                    condition={getCondition(conditionSource, conditionState)}
                    onSourceChange={handleConditionChange}
                    canEdit={canEdit}
                />

                {needTicket && requestApprovalVisible && !conditionApprovalError && !conditionApprovalResult && (
                    <div className="Condition-need-approval-button-area">
                        <Button
                            size="s"
                            view="action"
                            text={t("rule.requestApproval")}
                            onClick={handleRequestApproval}
                            disabled={conditionApprovalInProgress}
                        >
                            {t("rule.requestApproval")}
                        </Button>
                    </div>
                )}

                {needTicket && (
                    <div className="Condition-need-approval-button-area">
                        {conditionApprovalError && (
                            <span className="ConditionCard-request-approval-error">
                                {t("rule.requestApprovalFailed")}
                            </span>
                        )}
                        {conditionApprovalResult && (
                            <span className="ConditionCard-request-approval-result">
                                <Link
                                    href={`https://st.yandex-team.ru/${conditionApprovalResult.issueKey}`}
                                    target="_blank"
                                >
                                    {conditionApprovalResult.issueKey}
                                </Link>
                            </span>
                        )}
                    </div>
                )}
            </div>
        )
    );
});

export default Conditions;
