import React from "react";
import { connect } from "react-redux";
import { Button } from "@lib/components/lego";
import FontAwesome from "react-fontawesome";
import { map as _map, get as _get, uniq as _uniq } from "lodash";
import { useTranslation } from "react-i18next";
import AddConditionButton from "./AddConditionButton";
import { setActiveCondition } from "../actions";
import { getActiveConditionState, getRuleState } from "../selectors";

import "./style.scss";

const STATE_ICONS = {
    APPROVED: "thumbs-up",
    NEED_APPROVE: "clock-o",
};

const ConditionsSelector = connect((state) => ({
    rule: getRuleState(state),
    activeCondition: getActiveConditionState(state),
}))((props) => {
    const { conditions, rule, activeCondition, ruleId, canEdit, dispatch } = props;

    const { t } = useTranslation("condition");

    const updateCondition = (condition) => {
        let newActiveCondition = rule.conditions.find(
            (c) => c.source === condition.source && c.state === condition.state
        );

        dispatch(setActiveCondition(newActiveCondition));
    };

    const checked = (condition) => {
        return condition.source === activeCondition.source && condition.state === activeCondition.state;
    };

    return (
        <div className="ConditionsSelector">
            <div className="ConditionsSelector-bar">
                {_map(conditions, (each) => (
                    <span className="ConditionsSelector-button" key={each.source + "_" + each.state}>
                        <Button
                            view={checked(each) ? "default" : "pseudo"}
                            size="s"
                            onClick={() => updateCondition(each)}
                        >
                            {t("conditionSelector." + [each.source])} <ConditionsSelectorIcon condition={each} />
                        </Button>
                    </span>
                ))}
            </div>
            {canEdit && (
                <AddConditionButton
                    key="add-condition-button"
                    ruleId={ruleId}
                    sources={_uniq(_map(conditions, "source"))}
                    onSourceChange={props.onChange}
                />
            )}
        </div>
    );
});

const ConditionsSelectorIcon = (props) => {
    const { condition } = props;

    let className = "state-icon-" + condition.state.toLowerCase();
    let icon = _get(STATE_ICONS, condition.state, "question");

    if (condition.hasErrors) {
        className = "state-icon-error";
        icon = "exclamation-triangle";
    }

    return (
        <span className={className} title={condition.state}>
            <FontAwesome name={icon} />
        </span>
    );
};

export default ConditionsSelector;
