import React, { useEffect } from "react";
import { connect } from "react-redux";
import { useTranslation } from "react-i18next";
import FontAwesome from "react-fontawesome";
import Loading from "@lib/components/loading";
import Link from "@lib/components/link";
import { getExportInfoState, getSegmentByExportIdState } from "../selectors";
import Failable from "../../utils/Failable";
import State from "../card/State";
import { fetchSegmentByExportId } from "../actions";
import { getApiState } from "../../selectors";

import "./ExportTooltip.scss";

function getSegmentUrl(segmentId) {
    return `${location.origin}/segments?segment=${segmentId}`;
}

const ExportTooltip = connect((state) => ({
    api: getApiState(state),
    exportInfo: getExportInfoState(state).value,
    loading: getExportInfoState(state).loading,
    error: getExportInfoState(state).error,
    exportId: getExportInfoState(state).exportId,
    segmentByExportId: getSegmentByExportIdState(state).value,
    segmentLoading: getSegmentByExportIdState(state).loading,
}))((props) => {
    const { api, exportInfo, loading, error, exportId, segmentByExportId, segmentLoading, dispatch } = props;

    const { t } = useTranslation("segments");

    useEffect(() => {
        dispatch(fetchSegmentByExportId(api, exportId));
    }, [exportId]);

    return (
        <div className="ExportTooltip-container">
            <div className="ExportTooltip-modal-header">
                <span className="ExportTooltip-modal-header-export-id">{exportId}</span>
            </div>
            <div className="ExportTooltip-content">
                <Failable failed={error} errorMessage={t("exports.error")}>
                    <Loading loaded={!loading} size="l">
                        {exportInfo !== null && (
                            <table>
                                <tbody>
                                    <tr>
                                        <td className="ExportTooltip-label-segment-name">{t("exports.segmentName")}</td>
                                        <td className="ExportTooltip-value-segment-name">
                                            <Link href={getSegmentUrl(segmentByExportId.id)} target="blank">
                                                {!segmentLoading && segmentByExportId.name}
                                                <FontAwesome name="external-link" />
                                            </Link>
                                        </td>
                                    </tr>
                                    <tr>
                                        <td className="ExportTooltip-label">Keyword ID</td>
                                        <td className="ExportTooltip-value">{exportInfo.keywordId}</td>
                                    </tr>
                                    <tr>
                                        <td className="ExportTooltip-label">Segment ID</td>
                                        <td className="ExportTooltip-value">{exportInfo.segmentId}</td>
                                    </tr>
                                    <tr>
                                        <td className="ExportTooltip-label">Type</td>
                                        <td className="ExportTooltip-value">{exportInfo.type}</td>
                                    </tr>
                                    <tr>
                                        <td className="ExportTooltip-label">State</td>
                                        <td className="ExportTooltip-value">
                                            <State
                                                segmentId={exportInfo.segmentId}
                                                segmentExport={exportInfo}
                                                editable={false}
                                            />
                                        </td>
                                    </tr>
                                </tbody>
                            </table>
                        )}
                    </Loading>
                </Failable>
            </div>
        </div>
    );
});

export default ExportTooltip;
