import React, { useState } from "react";
import { connect, useDispatch } from "react-redux";
import { useTranslation } from "react-i18next";
import FontAwesome from "react-fontawesome";

import { Button, TextInput } from "@lib/components/lego";
import Id from "src/utils/Id";
import { getAccessState, getActiveIdsState, getApiState, getRuleState } from "src/selectors";
import { deleteSegmentExportRule, unlinkSegmentExportRule, updateRule } from "src/actions";
import ActionButton from "../utils/ActionButton";
import WarningIcon from "../utils/WarningIcon";
import DaysTooltip from "./DaysTooltip";
import RuleInfoCoverage from "./RuleInfoCoverage";
import UpdateEstimateStatsButton from "./UpdateEstimateStatsButton";

const RuleInfo = connect((state) => ({
    api: getApiState(state),
    isAdmin: getAccessState(state).Admin,
    segmentId: getActiveIdsState(state).segment,
}))((props) => {
    const { api, segmentId, exportId, rule, isAdmin, canEdit } = props;

    const dispatch = useDispatch();
    const { t, i18n } = useTranslation("segments");
    const lang = i18n.language;

    const handleDeleteRule = () => {
        const message = `[${rule.id}] Do you want to delete ${exportId} rule?`;
        if (window.confirm(message)) {
            dispatch(deleteSegmentExportRule(api, lang, segmentId, exportId));
        }
    };

    const handleUnlinkExportRule = () => {
        const message = `[${rule.id}] Do you want to unlink ${exportId} rule?`;
        if (window.confirm(message)) {
            dispatch(unlinkSegmentExportRule(api, lang, segmentId, exportId));
        }
    };

    return (
        <div>
            <table className="RuleInfo-table">
                <tbody>
                    <tr>
                        <td className="RuleInfo-label">ID</td>
                        <td>
                            <Id value={rule.id} copyable={true} />
                            {canEdit && (
                                <Button view="clear" theme="link" size="s" onClick={handleDeleteRule}>
                                    <FontAwesome name="trash" />
                                </Button>
                            )}
                        </td>
                    </tr>
                    <tr>
                        <td className="RuleInfo-label">{t("rule.usage")}</td>
                        <td>
                            <Id value={exportId} copyable={true} />
                            {isAdmin && (
                                <Button view="clear" theme="link" size="s" onClick={handleUnlinkExportRule}>
                                    <FontAwesome name="unlink" />
                                </Button>
                            )}
                        </td>
                    </tr>
                    <tr>
                        <td className="RuleInfo-label">{t("rule.userActivity")}</td>
                        <td>
                            <span className="RuleInfo-user-activity">
                                <span className="RuleInfo-user-activity-value">
                                    {rule.minDays + " / " + rule.days + " " + t("rule.days")}
                                </span>
                                {canEdit && <UsersActivityEditButton rule={rule} />}
                            </span>
                        </td>
                    </tr>
                    <tr>
                        <td className="RuleInfo-label">
                            {t("rule.estimate.dailyCoverage")}
                            <UpdateEstimateStatsButton />
                        </td>
                        <td>
                            <RuleInfoCoverage />
                        </td>
                    </tr>
                </tbody>
            </table>
        </div>
    );
});

const UsersActivityEditButton = connect((state) => ({
    api: getApiState(state),
    sources: getRuleState(state).conditions.map((condition) => condition.source),
}))((props) => {
    const { api, rule, sources } = props;

    const [isEditing, setIsEditing] = useState(false);
    const [days, setDays] = useState(rule.days);
    const [minDays, setMinDays] = useState(rule.minDays);
    const [daysWarning, setDaysWarning] = useState("");
    const [minDaysWarning, setMinDaysWarning] = useState("");

    const disabled = sources.length === 1 && sources[0] === "PRECALCULATED_TABLES";

    const initialize = () => {
        setIsEditing(false);
        setDays(undefined);
        setMinDays(undefined);
        setDaysWarning("");
        setMinDaysWarning("");
    };

    const dispatch = useDispatch();
    const { i18n } = useTranslation("segments");
    const lang = i18n.language;

    const updateRule_ = (api, lang, ruleId, name, days, minDays) =>
        dispatch(updateRule(api, lang, ruleId, name, days, minDays));

    const handleMinDaysChange = (value) => {
        const minDaysValue = value.slice(0, 2);
        const minDays = minDaysValue || props.rule.minDays;
        const days = days || props.rule.days;

        setMinDays(minDaysValue);
        setMinDaysWarning(getMinDaysWarning(minDays, days));
        setDaysWarning(getDaysWarning(minDays, days));
    };

    const handleDaysChange = (value) => {
        // TODO: constrain days between 1 and 35
        const daysValue = value.slice(0, 2);
        const minDays = minDays || props.rule.minDays;
        const days = daysValue || props.rule.days;

        setDays(daysValue);
        setDaysWarning(getDaysWarning(minDays, days));
        setMinDaysWarning(getMinDaysWarning(minDays, days));
    };

    const handleEditClick = () => {
        setIsEditing(!isEditing);
    };

    const handleAcceptEditing = () => {
        updateRule_(api, lang, rule.id, rule.name, days, minDays);

        initialize();
    };

    const handleRejectEditing = () => {
        initialize();
    };

    const getInvalidInputWarning = () => {
        return minDaysWarning || daysWarning;
    };

    const warning = getInvalidInputWarning();

    return (
        <span className="RuleInfo-edit-days-button" key={1}>
            <ActionButton
                key="edit-users-activity-button"
                action="edit"
                view="clear"
                theme="link"
                size="s"
                checked={isEditing}
                onClick={handleEditClick}
                disabled={disabled}
            />

            <DaysTooltip visible={isEditing}>
                <div>
                    <span className="DaysTooltip-item DayTooltip-days-input">
                        <TextInput
                            view="default"
                            size="m"
                            disabled={!isEditing}
                            error={minDaysWarning}
                            placeholder={rule.minDays.toString()}
                            value={minDays}
                            type="number"
                            onChange={(event) => handleMinDaysChange(event.target.value)}
                        />
                    </span>
                    <span className="DaysTooltip-item">/</span>
                    <span className="DaysTooltip-item DayTooltip-days-input">
                        <TextInput
                            error={daysWarning}
                            view="default"
                            size="m"
                            type="number"
                            disabled={!isEditing}
                            placeholder={rule.days.toString()}
                            value={days}
                            onChange={(event) => handleDaysChange(event.target.value)}
                        />
                    </span>
                    <span className="DaysTooltip-item">days</span>
                    <span className="DaysTooltip-item">
                        <Button view="default" size="m" disabled={warning !== ""} onClick={handleAcceptEditing}>
                            OK
                        </Button>
                    </span>

                    <span className="DaysTooltip-item">
                        <Button view="clear" size="m" onClick={handleRejectEditing}>
                            Close
                        </Button>
                    </span>
                </div>
                {warning && (
                    <div className="users-activity-warning-message">
                        <WarningIcon /> {warning}
                    </div>
                )}
            </DaysTooltip>
        </span>
    );
});

export function getMinDaysWarning(minDays, days) {
    if (isNaN(minDays)) {
        return "'minDays' value is not a number";
    } else if (parseInt(minDays) < 0) {
        return "'minDays' value is negative";
    } else if (parseInt(minDays) > 35) {
        return "'minDays' value is greater than 35";
    } else if (!isNaN(days) && parseInt(minDays) > parseInt(days)) {
        return "'minDays' value is greater than 'days' value";
    }
    return "";
}

export function getDaysWarning(minDays, days) {
    if (isNaN(days)) {
        return "'days' value is not a number";
    } else if (parseInt(days) < 0) {
        return "'days' value is negative";
    } else if (parseInt(days) > 35) {
        return "'days' value is greater than 35";
    } else if (!isNaN(minDays) && parseInt(minDays) > parseInt(days)) {
        return "'days' value is less than 'minDays' value";
    }
    return "";
}

export default RuleInfo;
