import React, { useEffect, useState, useCallback } from "react";
import { connect } from "react-redux";
import { isEmpty, debounce } from "lodash";
import { useTranslation } from "react-i18next";
import { Select, TextInput } from "@lib/components/lego";
import ValidableYtPath from "../../../utils/ValidableYtPath";
import IdTypeSelect from "../../../samples/IdTypeSelect";
import { setActiveColumn, validateTablePath } from "../../../utils/utilsActions";
import { getApiState, getSrcTableColumnsState } from "../../../selectors";
import SegmentFormPropertyTooltip from "../../../utils/tooltips";

import "./styles.scss";
import { isNil } from "lodash/lang";

const PrecalculatedTableCondition = connect((state) => ({
    api: getApiState(state),
    srcYtColumns: getSrcTableColumnsState(state).columns,
}))((props) => {
    const { values, api, srcYtColumns, editingMode, dispatch, updateConditionValues } = props;

    const [initialCondition, setInitialCondition] = useState({});
    const [currentCondition, setCurrentCondition] = useState({});

    const { t } = useTranslation("segments");

    const validateTableDelayed = useCallback(
        debounce((path) => dispatch(validateTablePath(api, path)), 500),
        []
    );

    const initializeCurrentValues = () => {
        // FIXME: do not return empty string from API
        if (values.length === 1 && !isEmpty(values[0])) {
            const currentValues = JSON.parse(values[0]);

            setInitialCondition(currentValues);
            setCurrentCondition(currentValues);
            dispatch(setActiveColumn(currentValues.idKey));
        }
    };

    useEffect(() => {
        initializeCurrentValues();
    }, [values]);

    const updateCondition = (key, value) => {
        setCurrentCondition({
            ...currentCondition,
            [key]: value,
        });
    };

    const updateColumn = (value) => {
        updateCondition("idKey", value);
        dispatch(setActiveColumn(value));
    };

    const updateUpdateInterval = (value) => {
        setCurrentCondition({
            ...currentCondition,
            updateInterval: value >= 0 ? value : 0,
        });
    };

    useEffect(() => {
        updateConditionValues([JSON.stringify(currentCondition)]);
    }, [currentCondition]);

    useEffect(() => {
        if (!isEmpty(currentCondition.path)) {
            validateTableDelayed(currentCondition.path);
        }
    }, [currentCondition.path]);

    useEffect(() => {
        if (isNil(initialCondition.idKey) && srcYtColumns.length > 0) {
            updateColumn(srcYtColumns[0].name);
        }
    }, [srcYtColumns]);

    return (
        <div className="PrecalculatedTableCondition">
            <table className="PrecalculatedTableParameters">
                <tbody>
                    <tr>
                        <td className="PrecalculatedTableLabel">{t("precalculatedTable.labels.path")}</td>
                        <td className="PrecalculatedTableValue">
                            <ValidableYtPath>
                                <TextInput
                                    view="default"
                                    theme="normal"
                                    size="s"
                                    hasClear
                                    value={currentCondition.path}
                                    placeholder={"//path/to/table"}
                                    onChange={(event) => updateCondition("path", event.target.value)}
                                    disabled={!editingMode}
                                />
                            </ValidableYtPath>
                        </td>
                        <td>
                            <SegmentFormPropertyTooltip text={t("precalculatedTable.tooltips.path")} />
                        </td>
                    </tr>
                    <tr>
                        <td className="PrecalculatedTableLabel">{t("precalculatedTable.labels.column")}</td>
                        <td className="PrecalculatedTableValue">
                            <Select
                                size="m"
                                view="default"
                                width="max"
                                onChange={(event) => updateColumn(event.target.value)}
                                value={currentCondition.idKey}
                                options={srcYtColumns.map((column) => ({
                                    value: column.name,
                                    content: column.name,
                                }))}
                                disabled={srcYtColumns.length === 0 || !editingMode}
                            />
                        </td>
                        <td>
                            <SegmentFormPropertyTooltip text={t("precalculatedTable.tooltips.column")} />
                        </td>
                    </tr>
                    <tr>
                        <td className="PrecalculatedTableLabel">{t("precalculatedTable.labels.idType")}</td>
                        <td className="PrecalculatedTableValue">
                            <IdTypeSelect
                                value={currentCondition.idType}
                                onChange={(value) => updateCondition("idType", value)}
                                disabled={!editingMode}
                            />
                        </td>
                        <td>
                            <SegmentFormPropertyTooltip text={t("precalculatedTable.tooltips.idType")} />
                        </td>
                    </tr>
                    <tr>
                        <td className="PrecalculatedTableLabel">{t("precalculatedTable.labels.updateInterval")}</td>
                        <td className="PrecalculatedTableValue">
                            <div className="PrecalculatedTableUpdatePeriodValue">
                                <TextInput
                                    className="PrecalculatedTableUpdatePeriod"
                                    view="default"
                                    theme="normal"
                                    size="s"
                                    type="number"
                                    value={currentCondition.updateInterval}
                                    onChange={(event) => updateUpdateInterval(event.target.value)}
                                    disabled={!editingMode}
                                />
                                <span className="PrecalculatedTableUpdatePeriodLabel">
                                    {t("precalculatedTable.labels.daysLabel")}
                                </span>
                            </div>
                        </td>
                        <td>
                            <SegmentFormPropertyTooltip text={t("precalculatedTable.tooltips.updateInterval")} />
                        </td>
                    </tr>
                </tbody>
            </table>
        </div>
    );
});

export default PrecalculatedTableCondition;
