import React, { useState } from "react";
import { connect } from "react-redux";
import FontAwesome from "react-fontawesome";
import { isEmpty, values as _values } from "lodash";
import { useTranslation } from "react-i18next";
import { Button, Modal, RadioButton, TextInput } from "@lib/components/lego";
import Link from "@lib/components/link";

import {
    getAccessState,
    getActiveIdsState,
    getActiveSegmentInfoState,
    getApiState,
    getSegmentsModalsState,
} from "../../selectors";
import { createSegment, createUserSegment, setCreateSegmentModalVisibility } from "../actions";
import { getCurrentUserLogin } from "src/utils/cookie";
import { DEFAULT_USER_DOMAIN } from "../../utils/constants";
import { getGroupsState } from "../selectors";
import LabModalHeader from "../../utils/LabModalHeader";

import "./style.scss";
import "./EditTreeButton.scss";
import "./CreateSegmentButton.scss";
import "./CreateSegmentForm.scss";

const PLACEHOLDERS = {
    name: "Мой сегмент",
    nameEn: "My segment",
    description: "Люди, которые ...",
    descriptionEn: "Persons who ...",
    tickets: "TASK-42",
};

const SEGMENT_TYPES = {
    cryptaSegment: "CRYPTA_SEGMENT",
    userSegment: "USER_SEGMENT",
    interest: "INTEREST",
};

const SCOPES = {
    internal: "INTERNAL",
    external: "EXTERNAL",
    externalPrivate: "EXTERNAL_PRIVATE",
};

const BUTTON_SIZE = "s";
const TEXT_INPUT_SIZE = "m";

const CreateSegmentButton = connect((state) => ({
    modalVisible: getSegmentsModalsState(state).createSegment,
    notAllowedSegments: getAccessState(state).RestrictedSegments,
}))((props) => {
    const { modalVisible, notAllowedSegments, dispatch } = props;

    const [pressed, setPressed] = useState(false);

    const { t } = useTranslation("segments");

    const openModal = () => {
        setPressed(true);
        dispatch(setCreateSegmentModalVisibility(true));
    };

    const closeModal = () => {
        setPressed(false);
        dispatch(setCreateSegmentModalVisibility(false));
    };

    // TODO: fix pressed display
    return [
        <Button
            key="create-segment-button"
            view="action"
            size="s"
            onClick={openModal}
            title={t("createForm.createButton")}
            checked={modalVisible}
        >
            <span className="small-screen-icon">
                <FontAwesome name="plus" />
            </span>
            <span className="normal-screen-icon">
                <FontAwesome name="plus" /> {t("createForm.createButton")}
            </span>
        </Button>,
        <div>
            {notAllowedSegments ? (
                <SegmentsNotAllowedDisclaimer onClose={closeModal} />
            ) : (
                <CreateSegmentModal key="create-segment-popup" onClose={closeModal} />
            )}
        </div>,
    ];
});

const SegmentsNotAllowedDisclaimer = connect((state) => ({
    visible: getSegmentsModalsState(state).createSegment,
}))((props) => {
    const { visible, onClose } = props;
    const { t } = useTranslation("segments");

    const surveysUrl = "https://wiki.yandex-team.ru/users/mohamad-mari/test";
    const supportChatUrl = "https://t.me/joinchat/GY4mgwUKuOKjNF4b";

    return (
        <Modal theme="normal" visible={visible} zIndex={1} onClose={onClose}>
            <div className="SegmentsNotAllowedDisclaimer">
                <LabModalHeader title={t("forbidden.header")} hasClose onClose={onClose} />
                <div className="SegmentsNotAllowedDisclaimer-body">
                    <div className="SegmentsNotAllowedDisclaimer-text">
                        {t("forbidden.external.start")}
                        <Link href={surveysUrl} target="_blank" tabindex={-1}>
                            {t("forbidden.external.end")}
                        </Link>
                    </div>
                    <div className="SegmentsNotAllowedDisclaimer-text">
                        {t("forbidden.internal.start")}
                        <Link href={supportChatUrl} target="_blank" tabindex={-1}>
                            {t("forbidden.internal.end")}
                        </Link>
                    </div>
                </div>
            </div>
        </Modal>
    );
});

const CreateSegmentModal = connect((state) => ({
    visible: getSegmentsModalsState(state).createSegment,
}))((props) => {
    const { visible, onClose } = props;

    const { t } = useTranslation("segments");

    return (
        <Modal theme="normal" visible={visible} zIndex={1} onClose={onClose}>
            <div className="CreateSegmentModal">
                <LabModalHeader title={t("createForm.creation")} hasClose onClose={onClose} />

                <div className="CreateSegmentModal-main">
                    <CreateSegmentBody visible={visible} />
                </div>
            </div>
        </Modal>
    );
});

const CreateSegmentBody = connect((state) => ({
    api: getApiState(state),
    isAdmin: getAccessState(state).Admin,
    activeSegmentId: getActiveIdsState(state).segment,
    activeSegmentInfo: getActiveSegmentInfoState(state),
    groups: getGroupsState(state),
}))((props) => {
    const { api, isAdmin, activeSegmentId, activeSegmentInfo, groups, dispatch } = props;

    const [name, setName] = useState("");
    const [nameEn, setNameEn] = useState("");
    const [warning, setWarning] = useState({
        name: true,
        description: true,
        tickets: [true],
    });
    const [description, setDescription] = useState("");
    const [descriptionEn, setDescriptionEn] = useState("");
    const [confirmedTickets, setConfirmedTickets] = useState([""]);
    const [segmentType, setSegmentType] = useState(SEGMENT_TYPES.cryptaSegment);
    const [scope, setScope] = useState(SCOPES.internal);
    const [confirmedResponsibles, setConfirmedResponsibles] = useState([
        {
            login: getCurrentUserLogin(),
            domain: DEFAULT_USER_DOMAIN,
        }, // yandex-team.ru
    ]);
    const [confirmedStakeholders, setConfirmedStakeholders] = useState([]);

    const { t, i18n } = useTranslation("segments");
    const lang = i18n.language;

    const getParentId = () => {
        if (activeSegmentId.startsWith("group") || activeSegmentId.startsWith("root")) {
            return activeSegmentId;
        } else {
            return activeSegmentInfo.value ? activeSegmentInfo.value.parentId : "root";
        }
    };
    const [parentId, setParentId] = useState(getParentId());

    const findGroupsName = (_parentId) => {
        let groups = groups ? groups.filter((group) => group.id === _parentId) : [];
        if (!isEmpty(groups)) {
            return groups[0].name;
        }

        return "";
    };
    const [parentName, setParentName] = useState(findGroupsName(parentId));

    const initialize = () => {
        setName("");
        setNameEn("");
        setDescription("");
        setDescriptionEn("");
        setConfirmedTickets([""]);
        setSegmentType(SEGMENT_TYPES.cryptaSegment);
        setScope(SCOPES.internal);
        setConfirmedResponsibles([
            {
                login: getCurrentUserLogin(),
                domain: DEFAULT_USER_DOMAIN,
            }, // yandex-team.ru
        ]);
        setConfirmedStakeholders([]);
        setParentId(getParentId());
        setParentName(findGroupsName());
        setWarning({
            name: [true],
            description: [true],
            tickets: [true],
        });
    };

    const handleNameChange = (value) => {
        setName(value);

        let updatedWarning = { ...warning };
        updatedWarning.name = isEmpty(value);
        setWarning(updatedWarning);
    };

    const handleDescriptionChange = (value) => {
        let newWarning = { ...warning };
        newWarning.description = isEmpty(value);

        setDescription(value);
        setWarning(newWarning);
    };

    const isTicketValid = (value) => {
        return value.match(/^([a-zA-Z]+-[1-9]+[0-9]*)$/);
    };

    const areTicketsValid = () => {
        return confirmedTickets.every((ticket) => isTicketValid(ticket));
    };

    const handleAddTicket = () => {
        let updatedTickets = [...confirmedTickets];
        updatedTickets.push("");

        let updatedWarning = { ...warning };
        updatedWarning.tickets.push(true);

        setConfirmedTickets(updatedTickets);
        setWarning(updatedWarning);
    };

    const handleChangeTicket = (index, value) => {
        let updatedTickets = [...confirmedTickets];
        updatedTickets[index] = value;

        let updatedWarning = { ...warning };
        updatedWarning.tickets[index] = !isTicketValid(value);

        setConfirmedTickets(updatedTickets);
        setWarning(updatedWarning);
    };

    const handleRemoveTicket = (index) => {
        let updatedTickets = [...confirmedTickets];
        updatedTickets.splice(index, 1);

        setConfirmedTickets(updatedTickets);
    };

    const handleChangeResponsibleLogin = (index, value) => {
        let updatedResponsibles = [...confirmedResponsibles];
        updatedResponsibles[index].login = value;

        setConfirmedResponsibles(updatedResponsibles);
    };

    const handleAddResponsible = () => {
        let updatedResponsibles = [...confirmedResponsibles];
        updatedResponsibles.push({ login: "", domain: DEFAULT_USER_DOMAIN });

        setConfirmedResponsibles(updatedResponsibles);
    };

    const handleRemoveResponsible = (index) => {
        let updatedResponsibles = [...confirmedResponsibles];
        updatedResponsibles.splice(index, 1);

        setConfirmedResponsibles(updatedResponsibles);
    };

    const handleChangeStakeholderLogin = (index, value) => {
        let updatedStakeholders = [...confirmedStakeholders];
        updatedStakeholders[index].login = value;

        setConfirmedStakeholders(updatedStakeholders);
    };

    const handleAddStakeholder = () => {
        let updatedStakeholders = [...confirmedStakeholders];
        updatedStakeholders.push({ login: "", domain: DEFAULT_USER_DOMAIN });

        setConfirmedStakeholders(updatedStakeholders);
    };

    const handleRemoveStakeholder = (index) => {
        let updatedStakeholders = [...confirmedStakeholders];
        updatedStakeholders.splice(index, 1);

        setConfirmedStakeholders(updatedStakeholders);
    };

    const isValidInput = () => {
        return name && description && !warning.name && !warning.description && areTicketsValid();
    };

    const handleCreateSegment = () => {
        if (isAdmin) {
            dispatch(
                createSegment(
                    api,
                    lang,
                    name,
                    nameEn,
                    description,
                    descriptionEn,
                    confirmedTickets,
                    segmentType,
                    scope,
                    parentId,
                    confirmedResponsibles.map((each) => each.login + "@" + DEFAULT_USER_DOMAIN),
                    confirmedStakeholders.map((each) => each.login + "@" + DEFAULT_USER_DOMAIN)
                )
            );
        } else {
            dispatch(createUserSegment(api, lang, name, nameEn, description, descriptionEn, confirmedTickets));
        }

        dispatch(setCreateSegmentModalVisibility(false));

        // TODO: indicate if not ok
        initialize();
    };

    return (
        <div className="CreateSegmentForm-container">
            <table className="CreateSegmentForm-table">
                <tbody>
                    <tr>
                        <td className="CreateSegmentForm-label">{t("createForm.name")}</td>
                        <td className="CreateSegmentForm-value">
                            <TextInput
                                state={warning.name ? "error" : undefined}
                                view="default"
                                size={TEXT_INPUT_SIZE}
                                hasClear
                                value={name}
                                placeholder={PLACEHOLDERS.name}
                                onChange={(event) => handleNameChange(event.target.value)}
                            />
                            {warning.name && (
                                <div className="CreateSegmentButton-warning">{t("createForm.alerts.name")}</div>
                            )}
                        </td>
                    </tr>
                    <tr>
                        <td className="CreateSegmentForm-label">{t("createForm.nameEn")}</td>
                        <td className="CreateSegmentForm-value">
                            <TextInput
                                view="default"
                                size={TEXT_INPUT_SIZE}
                                hasClear
                                value={nameEn}
                                placeholder={PLACEHOLDERS.nameEn}
                                onChange={(event) => setNameEn(event.target.value)}
                            />
                        </td>
                    </tr>
                    <tr>
                        <td className="CreateSegmentForm-label">{t("createForm.description")}</td>
                        <td className="CreateSegmentForm-value">
                            <TextInput
                                state={warning.description ? "error" : undefined}
                                view="default"
                                size={TEXT_INPUT_SIZE}
                                hasClear
                                value={description}
                                placeholder={PLACEHOLDERS.description}
                                onChange={(event) => handleDescriptionChange(event.target.value)}
                            />
                            {warning.description && (
                                <div className="CreateSegmentButton-warning">{t("createForm.alerts.description")}</div>
                            )}
                        </td>
                    </tr>

                    <tr>
                        <td className="CreateSegmentForm-label">{t("createForm.descriptionEn")}</td>
                        <td className="CreateSegmentForm-value">
                            <TextInput
                                view="default"
                                size={TEXT_INPUT_SIZE}
                                hasClear
                                value={descriptionEn}
                                placeholder={PLACEHOLDERS.descriptionEn}
                                onChange={(event) => setDescriptionEn(event.target.value)}
                            />
                        </td>
                    </tr>

                    {isAdmin && (
                        <tr>
                            <td className="CreateSegmentForm-label">{t("createForm.segmentType")}</td>
                            <td className="CreateSegmentForm-value">
                                <RadioButton
                                    value={segmentType}
                                    view="default"
                                    size="s"
                                    onChange={(event) => setSegmentType(event.target.value)}
                                    options={
                                        _values(SEGMENT_TYPES).map((each) => ({
                                            value: each,
                                            children: each,
                                        })) || []
                                    }
                                />
                            </td>
                        </tr>
                    )}

                    {isAdmin && (
                        <tr>
                            <td className="CreateSegmentForm-label">{t("createForm.scope")}</td>
                            <td className="CreateSegmentForm-value">
                                <RadioButton
                                    value={scope}
                                    view="default"
                                    size="s"
                                    onChange={(event) => setScope(event.target.value)}
                                    options={_values(SCOPES).map((each) => ({
                                        value: each,
                                        children: each,
                                    }))}
                                />
                            </td>
                        </tr>
                    )}

                    <tr>
                        <td className="CreateSegmentForm-label">{t("createForm.tickets")}</td>
                        <td className="CreateSegmentForm-value">
                            {confirmedTickets.map((ticket, index) => {
                                return (
                                    <div key={index}>
                                        <span className="CreateSegmentButton-tickets">
                                            <span className="CreateSegmentButton-ticket-value">
                                                <TextInput
                                                    state={warning.tickets[index] ? "error" : undefined}
                                                    view="default"
                                                    size={TEXT_INPUT_SIZE}
                                                    hasClear
                                                    value={ticket}
                                                    placeholder={PLACEHOLDERS.tickets}
                                                    onChange={(event) => handleChangeTicket(index, event.target.value)}
                                                />
                                            </span>

                                            <Button
                                                view="clear"
                                                theme="clear"
                                                size="s"
                                                onClick={() => handleRemoveTicket(index)}
                                            >
                                                <FontAwesome name="minus" />
                                            </Button>
                                        </span>

                                        {warning.tickets[index] && (
                                            <div className="CreateSegmentButton-warning">
                                                {t("createForm.alerts.ticket")}
                                            </div>
                                        )}
                                    </div>
                                );
                            })}

                            <div className="CreateSegmentButton-add-item">
                                <Button view="action" theme="action" size={BUTTON_SIZE} onClick={handleAddTicket}>
                                    {t("createForm.addTicket")}
                                </Button>
                            </div>
                        </td>
                    </tr>

                    <tr>
                        <td className="CreateSegmentForm-label">{t("createForm.responsibles")}</td>
                        <td className="CreateSegmentForm-value">
                            {confirmedResponsibles.map((responsible, index) => {
                                return (
                                    <div key={index} className="CreateSegmentButton-responsible">
                                        <span className="CreateSegmentButton-responsible-name">
                                            <TextInput
                                                view="default"
                                                size={TEXT_INPUT_SIZE}
                                                hasClear
                                                value={responsible.login}
                                                placeholder={""}
                                                width="100px"
                                                onChange={(event) =>
                                                    handleChangeResponsibleLogin(index, event.target.value)
                                                }
                                            />
                                        </span>

                                        <span className="CreateSegment-user-domain">@{DEFAULT_USER_DOMAIN}</span>

                                        <Button
                                            view="clear"
                                            theme="clear"
                                            size="s"
                                            onClick={() => handleRemoveResponsible(index)}
                                        >
                                            <FontAwesome name="minus" />
                                        </Button>
                                    </div>
                                );
                            })}

                            <div className="CreateSegmentButton-add-item">
                                <Button view="action" theme="action" size={BUTTON_SIZE} onClick={handleAddResponsible}>
                                    {t("createForm.addResponsible")}
                                </Button>
                            </div>
                        </td>
                    </tr>

                    <tr>
                        <td className="CreateSegmentForm-label">{t("createForm.stakeholders")}</td>
                        <td className="CreateSegmentForm-value">
                            {confirmedStakeholders.map((each, index) => {
                                return (
                                    <div key={index} className="CreateSegmentButton-steakholder">
                                        <span className="CreateSegmentButton-responsible-name">
                                            <TextInput
                                                view="default"
                                                size={TEXT_INPUT_SIZE}
                                                hasClear
                                                value={each.login}
                                                placeholder={""}
                                                width="100px"
                                                onChange={(event) =>
                                                    handleChangeStakeholderLogin(index, event.target.value)
                                                }
                                            />
                                        </span>

                                        <span className="CreateSegment-user-domain">@{DEFAULT_USER_DOMAIN}</span>

                                        <Button
                                            view="clear"
                                            theme="clear"
                                            size="s"
                                            onClick={() => handleRemoveStakeholder(index)}
                                        >
                                            <FontAwesome name="minus" />
                                        </Button>
                                    </div>
                                );
                            })}

                            <div className="CreateSegmentButton-add-item">
                                <Button view="action" theme="action" size={BUTTON_SIZE} onClick={handleAddStakeholder}>
                                    {t("createForm.addStakeholder")}
                                </Button>
                            </div>
                        </td>
                    </tr>

                    {isAdmin && (
                        <tr>
                            <td className="CreateSegmentForm-label">{t("createForm.parentId")}</td>
                            <td className="CreateSegmentForm-value">
                                <TextInput
                                    view="default"
                                    size={TEXT_INPUT_SIZE}
                                    hasClear
                                    value={parentId}
                                    placeholder={""}
                                    onChange={(event) => setParentId(event.target.value)}
                                    title={parentName}
                                />
                            </td>
                        </tr>
                    )}
                </tbody>
            </table>

            <div className="CreateSegmentForm-button">
                <Button
                    view="action"
                    theme="action"
                    size={BUTTON_SIZE}
                    disabled={!isValidInput()}
                    onClick={handleCreateSegment}
                >
                    {t("createForm.create")}
                </Button>
            </div>
        </div>
    );
});

export default CreateSegmentButton;
