import React, { useCallback, useEffect, useState } from "react";
import { connect } from "react-redux";
import { useTranslation } from "react-i18next";
import { debounce, filter as _filter, isEmpty } from "lodash";
import { TextInput, Spin } from "@lib/components/lego";
import { fetchSegmentsAttributes, matchSearchItems, resetSearch } from "../actions";

import {
    getActiveIdsState,
    getMatchedSearchItemsSegmentsState,
    getQueryState,
    getSegmentsInfoState,
} from "../selectors";
import { getApiState } from "../../selectors";
import { pushQuery, removeQuery } from "../../actions";

import "./SearchInput.scss";

const SearchInput = connect((state) => ({
    api: getApiState(state),
    activeGroupId: getActiveIdsState(state).group,
    loading: getMatchedSearchItemsSegmentsState(state).loading,
    query: getQueryState(state),
    segments: getSegmentsInfoState(state),
}))((props) => {
    const { activeGroupId, api, loading, query, segments, dispatch } = props;

    const [searchText, setSearchText] = useState("");

    const { t, i18n } = useTranslation("segments");

    const changeSearch = (value) => {
        const searchItems = _filter(value.split(" "));

        if (isEmpty(searchItems)) {
            dispatch(resetSearch());
            dispatch(removeQuery(["query"]));
            dispatch(fetchSegmentsAttributes(api, i18n.language));
        } else {
            dispatch(matchSearchItems(api, value.split(" ")));
            dispatch(pushQuery({ query: value }));
        }
    };

    const delayedSearch = useCallback(debounce(changeSearch, 500), []);

    const handleSearch_ = (value) => {
        setSearchText(value);
        delayedSearch(value);
    };

    useEffect(() => {
        if (!isEmpty(query)) {
            setSearchText(query);
            changeSearch(query);
        }
    }, [segments]);

    const setActiveGroupRoot = useCallback(() => {
        const newActive = {
            group: "root",
            segment: "root",
            export: undefined,
        };
        dispatch(pushQuery(newActive));
    }, [activeGroupId]);

    return (
        <>
            <TextInput
                className="SegmentsSearchInput-container"
                view="default"
                size="s"
                hasClear
                autoFocus
                value={searchText}
                placeholder={t("search.inputPlaceholder")}
                onChange={(event) => handleSearch_(event.target.value)}
                onClick={setActiveGroupRoot}
            />
            <span className="SegmentsSearchInput-loading">
                <Spin progress={loading} size="xs" view="default" />
            </span>
        </>
    );
});

export default SearchInput;
