import React, { useCallback, useEffect, useRef } from "react";
import { connect } from "react-redux";
import { isUndefined } from "lodash";
import FontAwesome from "react-fontawesome";
import { useTranslation } from "react-i18next";
import { getActiveIdsState } from "../selectors";
import { hasNotActiveSegmentExports, hasNoCoverage, hasNoExports, hasOverdue } from "../utils";

import SegmentTypePic from "./SegmentTypePic";
import MoveSegmentLabel from "./MoveSegmentLabel";
import NodeArrow from "./NodeArrow";

import "./SegmentItem.scss";
import { pushQuery } from "../../actions";

const SegmentItem = connect((state) => ({
    active: getActiveIdsState(state),
}))((props) => {
    const { active, segment, level, hasChildren, collapsed, onArrowClick, dispatch } = props;

    const { t } = useTranslation("segments");

    const myRef = useRef(null);

    useEffect(() => {
        if (isActive()) {
            myRef &&
                myRef.current &&
                myRef.current.scrollIntoView({
                    behavior: "smooth",
                    block: "center",
                    inline: "nearest",
                });
        }
    }, []);

    const onSegmentNameClick = useCallback(() => {
        const newActive = {
            ...active,
            segment: segment.id,
            export: undefined,
        };
        dispatch(pushQuery(newActive));
    }, [segment]);

    const isActive = () => {
        return segment.id === active.segment;
    };

    const isSegment = () => {
        return !isUndefined(segment.type);
    };

    const offset = 20;
    const captionPaddingShift = 40;
    const noArrowPaddingShift = hasChildren ? 0 : 20;
    let paddingLeft = `${level * offset + captionPaddingShift + noArrowPaddingShift}px`;

    return (
        <div
            ref={myRef}
            className={`SegmentItem${isActive() ? "-selected" : ""} ${
                segment.isSearched ? "SegmentItem-searched" : ""
            }`}
            onClick={onSegmentNameClick}
            style={{ paddingLeft: paddingLeft }}
        >
            {/*TODO node arrow only if visible*/}
            <NodeArrow visible={hasChildren} collapsed={collapsed} onClick={onArrowClick} />
            <div className="SegmentItem-text">
                <SegmentTypePic type={segment.type} />
                {segment.name}
                {segment.isSearched && <div className="SegmentItem-searched-icon" />}
            </div>
            <MoveSegmentLabel segment={segment} />

            <div className="SegmentItem-status pull-right">
                {isSegment() && hasOverdue(segment.exportStates) && (
                    <FontAwesome
                        title={t("item.tooltip.overdue")}
                        className="SegmentPic-overdue"
                        name="hourglass-end"
                    />
                )}

                {isSegment() &&
                    (hasNoExports(segment.exportStates) || hasNotActiveSegmentExports(segment.exportStates)) && (
                        <FontAwesome
                            title={t("item.tooltip.noExports")}
                            className="SegmentPic-empty"
                            name="times-circle"
                        />
                    )}

                {isSegment() && hasNoCoverage(segment.exportStates) && (
                    <FontAwesome
                        title={t("item.tooltip.noCoverage")}
                        className="SegmentPic-overdue pull-right"
                        name="exclamation-circle"
                    />
                )}
            </div>
        </div>
    );
});

export default SegmentItem;
