import React, { useState, useEffect, useCallback } from "react";
import { connect, useDispatch } from "react-redux";
import { invert as _invert, debounce, isEmpty, isUndefined } from "lodash";
import { useTranslation } from "react-i18next";

import { Button, TextInput, Select } from "@lib/components/lego";

import GroupSuggest from "./group-suggest";

import { fetchBaseSample, changeBaseType } from "../samples/actions";
import { getSamplesState } from "../samples/selectors";
import { getApiState } from "../selectors";

import "./style.scss";

const GLOBAL = "GLOBAL";
const SEGMENT = "SEGMENT";
const SAMPLE = "SAMPLE";

const BaseStatsParams = connect((state) => ({
    api: getApiState(state),
    baseSample: getSamplesState(state).baseSample,
    error: getSamplesState(state).errorBaseStats,
}))((props) => {
    const { api, baseSample, fetchStats, fetchStatsBySample, fetchStatsByExport, error } = props;

    const dispatch = useDispatch();

    const [baseType, setBaseType] = useState(GLOBAL);
    const [baseID, setBaseID] = useState("");
    const [baseGroup, setBaseGroup] = useState("");

    const [baseSubsamplesDict, setBaseSubsamplesDict] = useState({});

    const { t } = useTranslation("stats");

    const selectOptions = [
        { value: GLOBAL, content: t("describeType.global") },
        { value: SEGMENT, content: t("describeType.segment") },
        { value: SAMPLE, content: t("describeType.sample") },
    ];

    const updateBaseType = (newBaseType) => {
        if (newBaseType.target.value !== baseType) {
            setBaseType(newBaseType.target.value);
            setBaseID("");
            setBaseGroup("");
            dispatch(changeBaseType());
        }
    };

    const fetchBaseSampleDelayed = useCallback(
        debounce((baseID) => dispatch(fetchBaseSample(api, baseID)), 500),
        []
    );

    useEffect(() => {
        if (baseType === SAMPLE && !isEmpty(baseID)) {
            fetchBaseSampleDelayed(baseID);
        }
    }, [baseID]);

    useEffect(() => {
        setBaseSubsamplesDict(
            !isUndefined(baseSample.value) && !isEmpty(baseSample.value.userSetIdToGroupingKeyValue) ?
                _invert(baseSample.value.userSetIdToGroupingKeyValue) : {}
        );
    }, [baseSample.value]);

    const fetchAnyStats = () => {
        switch (baseType) {
            case GLOBAL:
                dispatch(fetchStats());
                break;
            case SEGMENT:
                dispatch(fetchStatsByExport(baseID));
                break;
            case SAMPLE:
                const baseUserSetId = !isEmpty(baseGroup) ? baseSubsamplesDict[baseGroup] : baseSample.value.siberiaUserSetId;
                dispatch(fetchStatsBySample(baseSample.value.id, baseGroup, baseUserSetId));
                break;
        }
    };

    return (
        <div className={"BaseSampleParams"}>
            <div className={"BaseSampleParams-parameters"}>
                <span>{t("baseSample")}</span>
                <Select
                    size="m"
                    view="default"
                    onChange={updateBaseType}
                    value={baseType}
                    options={selectOptions}
                    className="BaseSampleParams-select"
                />
                {
                    baseType !== GLOBAL && (
                        <TextInput
                            className="BaseSampleParams-input"
                            view="default"
                            theme="normal"
                            size="m"
                            hasClear
                            value={baseID}
                            placeholder={baseType === SAMPLE ? "Sample ID" : "Export ID"}
                            onChange={(event) => setBaseID(event.target.value)}
                        />
                    )
                }
                {
                    baseType === SAMPLE && !isEmpty(baseSubsamplesDict) && (
                        <div className="BaseSampleParams-group">
                            <GroupSuggest
                                sampleGroups={Object.keys(baseSubsamplesDict)}
                                isLoading={baseSample.isLoading}
                                activeGroup={baseGroup}
                                onSuggestionSelected={setBaseGroup}
                                onChange={setBaseGroup}
                            />
                        </div>
                    )
                }
                {(error || baseSample.error) && (<div className="BaseSampleParams-error">{t("baseNotFound")}</div>)}
            </div>
            <Button
                view="action"
                disabled={baseType !== GLOBAL && isEmpty(baseID)}
                className="BaseSampleParams-get-stats"
                onClick={fetchAnyStats}
            >
                {t("describe")}
            </Button>
        </div>
    );
});

export default BaseStatsParams;
