import store from "store";
import _ from "lodash";
import { createAction } from "redux-actions";
import { push } from "connected-react-router";
import { useState } from "react";

const MAX_EXTERNAL_SAMPLES_COUNT = 10;

export const requestTrainingSamples = createAction("request training samples");
export const receivedTrainingSamples = createAction("received training samples");
export const failedGettingTrainingSamples = createAction("failed getting training samples");

export const openCreateSampleModal = createAction("open create sample modal");
export const openUpdateSampleModal = createAction("open update sample modal");
export const closeCreateSampleModal = createAction("close create sample import modal");

export const openStatsModal = createAction("open stats model");
export const closeStatsModal = createAction("close stats model");

export const openMetricsModal = createAction("open metrics modal");
export const closeMetricsModal = createAction("close metrics modal");

export const changeTrainingSamplesType = createAction("change training sample type");

export function selectTrainingSamplesType(type) {
    return (dispatch) => {
        dispatch(selectTrainingSample(undefined));
        dispatch(changeTrainingSamplesType(type));
    };
}

function flattenSamples(samples) {
    let resultFetches = {
        my: [],
        viewed: [],
    };
    for (let samplesArray of samples) {
        for (let sampleItem of samplesArray) {
            if (sampleItem.sample) {
                resultFetches[sampleItem.type].push(sampleItem.sample);
            }
        }
    }

    return resultFetches;
}

export function getSampleById(samples, sampleId) {
    return Object.values(samples).filter((each) => each.id === sampleId)[0];
}

export function fetchTrainingSamples(state) {
    return (dispatch) => {
        dispatch(requestTrainingSamples());

        const { api } = state;
        let lastSeen = store.get("last_seen_training_samples", {});

        if (state.selected) {
            lastSeen[state.selected] = Date.now();
        }

        let fetches = [];
        fetches.push(
            api
                .then((api) => api.apis.lab.getTrainingSamples)
                .then((method) => method({ lang: state.lang }))
                .then((response) =>
                    _(response.obj)
                        .sortBy((sample) => sample["timestamps"]["modified"])
                        .map((each) => ({ type: "my", sample: each }))
                        .reverse()
                        .value()
                )
        );
        fetches.push(
            ..._(lastSeen)
                .toPairs()
                .orderBy((each) => each[1])
                .slice(-MAX_EXTERNAL_SAMPLES_COUNT)
                .reverse()
                .map((pair) =>
                    api
                        .then((api) => api.apis.lab.getTrainingSample({ id: pair[0] }))
                        .then((response) => response.obj)
                        .catch((error) => null)
                )
                .map((each) => each.then((value) => [{ type: "viewed", sample: value }]))
        );
        return Promise.all(fetches)
            .then(flattenSamples)
            .then((samples) => {
                let myIds = new Set(samples.my.map((x) => x.id));
                samples.viewed = samples.viewed.filter((x) => !myIds.has(x.id));
                let viewedIds = new Set(samples.viewed.map((x) => x.id));

                lastSeen = _.pickBy(lastSeen, (v, k) => viewedIds.has(k));
                store.set("last_seen_training_samples", lastSeen);

                let allSamples = samples.my.concat(samples.viewed);
                let resultingSamples = {
                    forNewModel: [],
                    toUpdateModel: [],
                };
                for (let sample of allSamples) {
                    if (_.isEmpty(sample.modelName)) {
                        resultingSamples["forNewModel"].push(sample);
                    } else {
                        resultingSamples["toUpdateModel"].push(sample);
                    }
                }

                let samplesSortFunc = (lhs, rhs) => rhs.timestamps.created - lhs.timestamps.created;
                resultingSamples.forNewModel.sort(samplesSortFunc);
                resultingSamples.toUpdateModel.sort(samplesSortFunc);

                if (state.selected) {
                    for (let sampleType in resultingSamples) {
                        if (resultingSamples[sampleType].filter((sample) => sample.id === state.selected).length) {
                            dispatch(changeTrainingSamplesType(sampleType));
                            break;
                        }
                    }
                }

                return resultingSamples;
            })
            .then((data) => {
                dispatch(receivedTrainingSamples(data));
            })
            .catch((error) => dispatch(failedGettingTrainingSamples(error)));
    };
}

export function selectTrainingSample(id) {
    return (dispatch) => {
        dispatch(push({ search: "?sample=" + id }));
    };
}

export function selectAndOpenStatsModal(id) {
    return (dispatch) => {
        dispatch(openStatsModal());
        dispatch(selectTrainingSample(id));
    };
}

export function selectAndOpenMetricsModal(id) {
    return (dispatch) => {
        dispatch(openMetricsModal());
        dispatch(selectTrainingSample(id));
    };
}

export const useFetching = (api, callback, defaultState) => {
    const [isLoading, setIsLoading] = useState(defaultState.isLoading);
    const [isFailed, setIsFailed] = useState(defaultState.isFailed);
    const [error, setError] = useState(defaultState.error);
    const [data, setData] = useState(defaultState.data);

    const fetching = (...args) => {
        setIsLoading(true);
        setIsFailed(false);

        api.then(callback)
            .then((method) => method(...args))
            .then((response) => {
                if (response.status === 204) {
                    throw new Error("Not found");
                }
                setData(response.obj);
            })
            .catch((error) => {
                setError(error.message);
                setIsFailed(true);
            })
            .finally(() => setIsLoading(false));
    };

    return [fetching, isLoading, isFailed, error, data];
};
