import React, { Component } from "react";
import { withTranslation } from "react-i18next";
import { connect } from "react-redux";
import { debounce, isEmpty } from "lodash";
import { Button, Modal } from "@lib/components/lego";
import Link from "@lib/components/link";
import LabModalHeader from "../../utils/LabModalHeader";
import { resetYtTableColumns, validateTablePath } from "../../utils/utilsActions";
import { getAccessState, getApiState, getLanguageState, getSrcTableColumnsState } from "../../selectors";
import { getSelectedTrainingSampleId, getTrainingSamplesState } from "../selectors";
import {
    changeTrainingSamplesType,
    closeCreateSampleModal,
    fetchTrainingSamples,
    selectTrainingSample,
} from "../actions";
import ImportTrainingSampleParams from "./SampleParameters";

import helpPic from "../../images/help.svg";
import "../styles/CreateSampleModal.scss";

const ACCESS_LEVEL_PLACEHOLDER = "SHARED";
const INPUT_TYPE_PLACEHOLDER = "TABLE";
const LABORATORY_TRAINING_SAMPLES_HELP_URL = "https://docs.yandex-team.ru/crypta/lab/training_samples";
const AUDIENCE_SEGMENT = "audienceSegment";
const PATH = "path";
const FILE = "file";
const NAME = "name";
const POSITIVE_SEGMENT_SIZE = "positiveSegmentSize";
const NEGATIVE_SEGMENT_SIZE = "negativeSegmentSize";
const TTL = "ttl";
const ACCESS_LEVEL = "accessLevel";
const INPUT_TYPE = "inputType";
const PARTNER = "partner";
const MODEL_NAME = "modelName";
const LOGINS_TO_SHARE = "loginsToShare";

@withTranslation("createSample")
class CreateTrainingSampleModal extends Component {
    constructor(props) {
        super(props);
        this.handleParameterChange = this.handleParameterChange.bind(this);
        this.handleHide = this.handleHide.bind(this);

        this.validateTablePathThrottled = debounce(props.validateTablePath, 1000);

        this.state = {
            [NAME]: "",
            [POSITIVE_SEGMENT_SIZE]: "",
            [NEGATIVE_SEGMENT_SIZE]: "",
            [TTL]: "",
            [ACCESS_LEVEL]: ACCESS_LEVEL_PLACEHOLDER,
            [PATH]: "",
            [FILE]: undefined,
            [AUDIENCE_SEGMENT]: "",
            [INPUT_TYPE]: INPUT_TYPE_PLACEHOLDER,
            [PARTNER]: undefined,
            [MODEL_NAME]: undefined,
            [LOGINS_TO_SHARE]: "",

            allFieldsDisabled: false,

            isSubmitted: false,
            isLoading: false,
            isFailed: false,
            error: null,
        };
    }

    componentDidMount() {
        this.parameters = new Set([
            PATH,
            FILE,
            AUDIENCE_SEGMENT,
            NAME,
            POSITIVE_SEGMENT_SIZE,
            NEGATIVE_SEGMENT_SIZE,
            TTL,
            ACCESS_LEVEL,
            MODEL_NAME,
            LOGINS_TO_SHARE,
            PARTNER,
        ]);
    }

    handleParameterChange(value, type) {
        switch (type) {
            case NAME:
                this.setState({ [NAME]: value });
                break;
            case INPUT_TYPE:
                this.setState({ [INPUT_TYPE]: value });
                break;
            case TTL:
                this.setState({ [TTL]: value });
                break;
            case ACCESS_LEVEL:
                this.setState({ [ACCESS_LEVEL]: value });
                break;
            case MODEL_NAME:
                this.setState({ [MODEL_NAME]: value });
                break;
            case PARTNER:
                this.setState({ [PARTNER]: value });
                break;
            case POSITIVE_SEGMENT_SIZE:
                this.setState({ [POSITIVE_SEGMENT_SIZE]: value });
                break;
            case NEGATIVE_SEGMENT_SIZE:
                this.setState({ [NEGATIVE_SEGMENT_SIZE]: value });
                break;
            case PATH:
                this.setState({ [PATH]: value, [FILE]: undefined, [AUDIENCE_SEGMENT]: "" });
                this.validateTablePathThrottled(this.props.api, value);
                break;
            case FILE:
                this.setState({ [FILE]: value, [PATH]: "", [AUDIENCE_SEGMENT]: "" });
                break;
            case AUDIENCE_SEGMENT:
                this.setState({ [AUDIENCE_SEGMENT]: value, [PATH]: "", [FILE]: undefined });
                break;
            case LOGINS_TO_SHARE:
                this.setState({ [LOGINS_TO_SHARE]: value });
                break;
            default:
                console.log("Undefined type");
                break;
        }
    }

    createTrainingSample() {
        this.setState({
            isSubmitted: true,
            isLoading: true,
            isFailed: false,
            allFieldsDisabled: true,
            error: null,
        });

        const parametersValues = Object.fromEntries(
            Object.entries(this.state)
                .filter((item) => this.parameters.has(item[0]) && item[1])
                .map((item) => {
                    if (item[0] === TTL) {
                        item[1] = Number(item[1]) * 24 * 60 * 60;
                    } else if (item[0] === POSITIVE_SEGMENT_SIZE || item[0] === NEGATIVE_SEGMENT_SIZE) {
                        item[1] = Number(item[1]);
                    }
                    return item;
                })
        );

        this.props.api
            .then((api) => {
                return parametersValues.path || parametersValues.audienceSegment
                    ? api.apis.lab.createTrainingSampleExtended
                    : api.apis.lab.createTrainingSampleFromFile;
            })
            .then((method) => method(parametersValues))
            .then((response) => {
                // add sample to search line (also redefine selected (sample))
                this.props.dispatch(selectTrainingSample(response.obj.id));
            })
            .catch((error) => {
                this.setState({
                    isFailed: true,
                    error: error.message,
                });
            })
            .finally(() =>
                this.setState({
                    isLoading: false,
                    allFieldsDisabled: false,
                })
            );
    }

    // hide after OK message and onClose
    handleHide() {
        this.props.dispatch(resetYtTableColumns());
        this.props.dispatch(closeCreateSampleModal());
    }

    handleClose() {
        this.handleHide();
        this.props.dispatch(
            fetchTrainingSamples({
                api: this.props.api,
                lang: this.props.lang,
                selected: this.props.selected,
            })
        );
        this.props.dispatch(changeTrainingSamplesType(this.props.isNewModel ? "forNewModel" : "toUpdateModel"));
    }

    render() {
        const { visible, lang, isAdmin, srcTableValid, t } = this.props;

        return (
            <div className="CreateSample">
                <Modal theme="normal" visible={visible} onClose={this.handleHide}>
                    <LabModalHeader
                        title={this.props.isNewModel ? t("creationTrainingSample") : t("updateExistingModel")}
                        info={
                            <Link
                                className="TrainingSampleCreationModal-help-link"
                                href={LABORATORY_TRAINING_SAMPLES_HELP_URL}
                                title={t("userGuide")}
                                target="_blank"
                            >
                                <img alt="" className="TrainingSampleCreationModal-help-pic" src={helpPic} />
                            </Link>
                        }
                        hasClose
                        onClose={this.handleHide}
                    />
                    <div className="TrainingSampleCreationModal-body">
                        <div className="TrainingSampleCreationModal-content">
                            <ImportTrainingSampleParams
                                lang={lang}
                                isNewModel={this.props.isNewModel}
                                onParameterChange={(value, type) => this.handleParameterChange(value, type)}
                                api={this.props.api}
                                inputType={this.state.inputType}
                                path={this.state.path}
                                audienceSegment={this.state.audienceSegment}
                                file={this.state.file}
                                name={this.state.name}
                                modelName={this.state.modelName}
                                partner={this.state.partner}
                                positiveSegmentSize={this.state.positiveSegmentSize}
                                negativeSegmentSize={this.state.negativeSegmentSize}
                                ttl={this.state.ttl}
                                loginsToShare={this.state.loginsToShare}
                                accessLevel={this.state.accessLevel}
                                disabled={this.state.allFieldsDisabled}
                                isAdmin={isAdmin}
                            />
                        </div>
                        {(!this.state.isSubmitted || this.state.isFailed) && (
                            <div className="SubmitButton">
                                <Button
                                    view="action"
                                    theme="action"
                                    size="s"
                                    onClick={() => this.createTrainingSample()}
                                    disabled={
                                        this.state.allFieldsDisabled ||
                                        (!srcTableValid && !this.state.audienceSegment && !this.state.file) ||
                                        !this.state.name ||
                                        (!this.props.isNewModel && (!this.state.partner || !this.state.modelName))
                                    }
                                >
                                    {t("create")}
                                </Button>
                            </div>
                        )}
                        {this.state.isSubmitted && (
                            <div className="TrainingSampleCreationModal-status">
                                {this.state.isLoading ? (
                                    <div className="TrainingSampleCreationModal-status-loading">Loading</div>
                                ) : (
                                    <div>
                                        {this.state.isFailed ? (
                                            <div className="TrainingSampleCreationModal-status-error">
                                                {this.state.error}
                                            </div>
                                        ) : (
                                            <div>
                                                <div className="TrainingSampleCreationModal-status-success">
                                                    Success
                                                </div>
                                                <div className="CloseButton">
                                                    <Button
                                                        view="action"
                                                        theme="action"
                                                        size="s"
                                                        onClick={() => this.handleClose()}
                                                    >
                                                        {t("close")}
                                                    </Button>
                                                </div>
                                            </div>
                                        )}
                                    </div>
                                )}
                            </div>
                        )}
                    </div>
                </Modal>
            </div>
        );
    }
}

function mapStateToProps(state) {
    return {
        api: getApiState(state),
        srcYtColumns: getSrcTableColumnsState(state).columns,
        srcTableValid: !isEmpty(getSrcTableColumnsState(state).columns),
        selected: getSelectedTrainingSampleId(state),
        isNewModel: getTrainingSamplesState(state).modals.newModel,
        visible: getTrainingSamplesState(state).modals.createSample,
        isAdmin: getAccessState(state).Admin,
        lang: getLanguageState(state),
    };
}

function mapDispatchToProps(dispatch) {
    return {
        validateTablePath: (api, path) => dispatch(validateTablePath(api, path)),
        resetYtTableColumns: () => dispatch(resetYtTableColumns()),
        dispatch,
    };
}

export default connect(mapStateToProps, mapDispatchToProps)(CreateTrainingSampleModal);
