import React, { useEffect } from "react";
import { connect } from "react-redux";
import { useTranslation } from "react-i18next";
import { Modal } from "@lib/components/lego";
import Loading from "@lib/components/loading";
import Link from "@lib/components/link";
import Failable from "../../utils/Failable";
import { getApiState, getEnvTypeState } from "../../selectors";
import { getSelectedTrainingSample, getTrainingSamplesState } from "../selectors";
import { closeMetricsModal, useFetching } from "../actions";

import sandboxPic from "src/images/sandbox.png";
import YtPathLink from "../../utils/YtPath";

import "../styles/Metrics.scss";

const MetricsModal = connect((state) => ({
    api: getApiState(state),
    envType: getEnvTypeState(state),
    isMetricsModalVisible: getTrainingSamplesState(state).modals.showMetrics,
    selectedSample: getSelectedTrainingSample(state),
}))((props) => {
    const { api, envType, isMetricsModalVisible, selectedSample, dispatch } = props;
    const { t } = useTranslation("trainingSamples");

    const [metricsFetching, isLoading, isFailed, error, metrics] = useFetching(
        api,
        (api) => api.apis.lab.getTrainingSampleMetrics,
        {
            isLoading: true,
            isFailed: false,
            error: undefined,
            data: undefined,
        }
    );

    useEffect(() => {
        metricsFetching({ id: selectedSample.id });
    }, [selectedSample]);

    useEffect(() => {
        metricsFetching({ id: selectedSample.id });
    }, []);

    const YT_BASE_URL = "//home/crypta/" + envType + "/lab/samples/training_samples/" + selectedSample.id;

    const renderMetrics = () => {
        return (
            <div className="MetricsModal-body">
                <div className="MetricsModal-body-left">
                    <div className="MetricsModal-body-data">
                        <div className="MetricsModal-body-data-header">{t("metrics.data")}</div>
                        <table className="MetricsModal-body-data-table">
                            <tbody>
                                <tr key="scores" className="MetricsModal-data-row">
                                    <td>{t("metrics.usersScores")}</td>
                                    <td>
                                        <YtPathLink
                                            tablePath={`${YT_BASE_URL}/resulting_segments`}
                                            linkTitle={t("goToYt")}
                                        />
                                    </td>
                                </tr>
                                <tr key="extended-metrics" className="MetricsModal-data-row">
                                    <td>{t("metrics.extended")}</td>
                                    <td>
                                        <YtPathLink tablePath={`${YT_BASE_URL}/metrics`} linkTitle={t("goToYt")} />
                                    </td>
                                </tr>
                            </tbody>
                        </table>
                    </div>
                    <div className="MetricsModal-body-metrics">
                        <table key="body-metrics-values">
                            <thead>
                                <tr key="table-header" className="MetricsModal-table-header">
                                    <td>{t("metrics.name")}</td>
                                    <td>{t("metrics.value")}</td>
                                </tr>
                            </thead>
                            <tbody>
                                <tr key="roc-auc" className="MetricsModal-table-row">
                                    <td>Roc-Auc</td>
                                    <td>{Math.round(metrics.rocAuc * 100) / 100}</td>
                                </tr>
                                <tr key="accuracy" className="MetricsModal-table-row">
                                    <td>Accuracy</td>
                                    <td>{Math.round(metrics.accuracy * 100) / 100}</td>
                                </tr>
                                <tr key="matched-ids-ratio" className="MetricsModal-table-row">
                                    <td>{t("metrics.matchedIds")}</td>
                                    <td>{Math.round(metrics.matchedIdsRatio * 100) / 100}</td>
                                </tr>
                                <tr key="positive-class-ratio" className="MetricsModal-table-row">
                                    <td>{t("metrics.positiveClassRatio")}</td>
                                    <td>{Math.round(metrics.positiveClassRatio * 100) / 100}</td>
                                </tr>
                                <tr key="negative-class-ratio" className="MetricsModal-table-row">
                                    <td>{t("metrics.negativeClassRatio")}</td>
                                    <td>{Math.round(metrics.negativeClassRatio * 100) / 100}</td>
                                </tr>
                            </tbody>
                        </table>
                    </div>
                </div>
                <div className="MetricsModal-body-features">
                    <div className="MetricsModal-body-features-name">{t("metrics.topFeatures")}</div>
                    <ul>
                        {metrics.topFeatures.map((each) => {
                            return <li key={each}>{each}</li>;
                        })}
                    </ul>
                </div>
            </div>
        );
    };

    const SANDBOX_TASK_BASE_URL = "https://sandbox.yandex-team.ru/tasks/?tags=";

    return (
        <Modal
            theme="normal"
            visible={isMetricsModalVisible}
            hasAnimation={false}
            onClose={() => dispatch(closeMetricsModal())}
        >
            <div className="MetricsModal">
                <div key="header" className="MetricsModal-header">
                    <span key="name" className="MetricsModal-name">
                        {selectedSample.name}
                    </span>

                    <div key="header" className="MetricsModal-header-right">
                        <Link href={SANDBOX_TASK_BASE_URL + selectedSample.id} target="_blank">
                            <img alt="Sandbox" className="SandboxLink" src={sandboxPic} />
                        </Link>
                        <span key="close" className="MetricsModal-close" onClick={() => dispatch(closeMetricsModal())}>
                            ⨉
                        </span>
                    </div>
                </div>
                <Loading loaded={!isLoading}>
                    <Failable failed={isFailed} errorMessage={error}>
                        {metrics && renderMetrics()}
                    </Failable>
                </Loading>
            </div>
        </Modal>
    );
});

export default MetricsModal;
