import React, { useEffect, useState } from "react";
import { connect } from "react-redux";
import { useTranslation } from "react-i18next";
import FontAwesome from "react-fontawesome";
import { Modal } from "@lib/components/lego";
import Loading from "@lib/components/loading";
import SampleStats from "@lib/stats/SampleStats";
import Failable from "../../utils/Failable";
import DefaultErrorPage from "../../utils/Errors";
import YtPathLink from "../../utils/YtPath";
import { getApiState, getEnvTypeState } from "../../selectors";
import { getSelectedTrainingSample, getTrainingSamplesState } from "../selectors";
import { closeStatsModal, useFetching } from "../actions";
import SegmentSelector from "./SegmentSelector";

import "../styles/SegmentsStats.scss";

const SegmentsStatsModal = connect((state) => ({
    api: getApiState(state),
    envType: getEnvTypeState(state),
    isStatsModalVisible: getTrainingSamplesState(state).modals.showStats,
    selectedSample: getSelectedTrainingSample(state),
}))((props) => {
    const { api, envType, isStatsModalVisible, selectedSample, dispatch } = props;

    const { t } = useTranslation("trainingSamples");

    const [activeSegment, setActiveSegment] = useState(undefined);
    const [segmentsFetching, isLoading, isFailed, error, segments] = useFetching(
        api,
        (api) => api.apis.lab.getTrainableSegments,
        {
            isLoading: true,
            isFailed: false,
            error: undefined,
            data: undefined,
        }
    );
    const [statsFetching, isStatsLoading, isGettingStatsFailed, errorGettingStats, stats] = useFetching(
        api,
        (api) => api.apis.lab.getStatsFromSiberia,
        {
            isLoading: true,
            isFailed: false,
            error: undefined,
            data: undefined,
        }
    );

    const hasSegments = () => (segments || []).length > 0;

    const onSegmentSelect = (segmentName) => {
        setActiveSegment(segmentName);

        let segment = Object.values(segments).filter((each) => each.name === segmentName)[0];

        statsFetching({ user_set_id: segment.siberiaUserSetId });
    };

    useEffect(() => {
        setActiveSegment("");
        if (selectedSample.id) {
            segmentsFetching({ id: selectedSample.id });
        }
    }, [selectedSample.id]);

    const YT_BASE_URL = "//home/crypta/" + envType + "/lab/samples/training_samples/" + selectedSample.id + "/segments";

    return (
        <Modal
            theme="normal"
            visible={isStatsModalVisible}
            hasAnimation={false}
            onClose={() => dispatch(closeStatsModal())}
        >
            {selectedSample && (
                <div className="SegmentsStatsModal">
                    <div key="header" className="SegmentsStatsModal-header">
                        <span key="name" className="SegmentsStatsModal-segment-name">
                            {selectedSample.name}
                        </span>
                        <div key="middle">
                            {hasSegments() && (
                                <div className="SegmentsStatsModal-middle">
                                    <span key="selector" className="SegmentsStatsModal-selector">
                                        <SegmentSelector
                                            key="segments"
                                            activeSegment={activeSegment}
                                            onSegmentSelect={onSegmentSelect}
                                        />
                                    </span>
                                    {activeSegment && (
                                        <span key="link" className="SegmentsStatsModal-table-link">
                                            <YtPathLink
                                                tablePath={`${YT_BASE_URL}/${activeSegment}`}
                                                linkTitle={t("goToYt")}
                                            />
                                        </span>
                                    )}
                                </div>
                            )}
                        </div>
                        <span className="SegmentsStatsModal-close" onClick={() => dispatch(closeStatsModal())}>
                            ⨉
                        </span>
                    </div>
                    <div key="main" className="SegmentsStatsModal-body">
                        <Loading loaded={!isLoading}>
                            <Failable failed={isFailed} errorMessage={error}>
                                {!activeSegment && (
                                    <div>
                                        {hasSegments() ? (
                                            <div className="SegmentsStatsModal-tip">
                                                <FontAwesome name="angle-double-up" />
                                                <div>{t("stats.selectSegmentTip")}</div>
                                            </div>
                                        ) : (
                                            <div className="SegmentsStatsModal-no-segments">
                                                <DefaultErrorPage errorMessage={t("stats.noSegmentsError")} />
                                            </div>
                                        )}
                                    </div>
                                )}
                            </Failable>
                        </Loading>
                        {activeSegment && (
                            <Loading loaded={!isStatsLoading}>
                                <Failable failed={isGettingStatsFailed} errorMessage={errorGettingStats}>
                                    <div key="body-values" className="SegmentsStatsModal-content">
                                        {stats && (
                                            <SampleStats
                                                loaded={!isStatsLoading}
                                                stats={{
                                                    value: stats.Stats,
                                                    ready: stats.Info.Ready,
                                                }}
                                                exportFileBasename={activeSegment}
                                                withLoading={false}
                                            />
                                        )}
                                    </div>
                                </Failable>
                            </Loading>
                        )}
                    </div>
                </div>
            )}
        </Modal>
    );
});

export default SegmentsStatsModal;
