import React, { Component } from "react";
import _ from "lodash";
import { scaleLinear, scaleBand, scaleOrdinal } from "d3-scale";
import { schemeSet1 } from "d3-scale-chromatic";

const COLORS = scaleOrdinal(schemeSet1);

class BarChart extends Component {
    render() {
        const ys = _(this.props.values).map("y");
        const xs = _(this.props.values).map("x");
        const sum = ys.sum();
        const margin = this.props.width / 3;
        const topMargin = 30;
        const yScale = scaleLinear()
            .domain([ys.min(), ys.max()])
            .range([1, this.props.width - 2 * margin]);
        const xScale = scaleBand()
            .domain(xs.value())
            .range([topMargin, this.props.height])
            .paddingInner(0.01)
            .paddingOuter(0.05);
        const bandwidth = xScale.bandwidth();
        return (
            <svg width={this.props.width} height={this.props.height}>
                <text
                    x={margin}
                    y={topMargin / 2}
                    className="Histogram-header"
                    textAnchor="middle"
                    alignmentBaseline="middle"
                >
                    {this.props.label}
                </text>
                {_(this.props.values)
                    .map((each, i) => {
                        return (
                            <rect
                                key={"bar" + i}
                                fill={COLORS(this.props.label)}
                                y={xScale(each.x)}
                                x={margin}
                                height={bandwidth}
                                width={yScale(each.y)}
                            />
                        );
                    })
                    .value()}
                {_(this.props.values)
                    .map((each, i) => {
                        return (
                            <text
                                key={"label" + i}
                                y={xScale(each.x) + bandwidth / 2}
                                x={margin - 2}
                                textAnchor="end"
                                alignmentBaseline="middle"
                            >
                                {each.x}
                            </text>
                        );
                    })
                    .value()}
                {_(this.props.values)
                    .map((each, i) => {
                        return (
                            <text
                                key={"value" + i}
                                y={xScale(each.x) + bandwidth / 2}
                                x={margin + yScale(each.y) + 2}
                                className="Histogram-value"
                                textAnchor="left"
                                alignmentBaseline="middle"
                            >
                                {((100 * each.y) / sum).toFixed(2)}%
                            </text>
                        );
                    })
                    .value()}
            </svg>
        );
    }
}

class Histogram extends Component {
    render() {
        const values = _(this.props.data)
            .map((obj) => {
                return { y: _.get(obj, "Count"), x: _.get(obj, this.props.x) };
            })
            .sortBy("x")
            .value();
        return (
            <div className="Histogram">
                <BarChart label={this.props.label} values={values} width={200} height={250} />
            </div>
        );
    }
}

class Affinities extends Component {
    render() {
        const tokens = this.props.data.Token;
        if (_.isEmpty(tokens)) {
            return null;
        }
        const values = _(tokens)
            .map((each) => {
                return { x: each.Token, y: each.Weight / each.Count };
            })
            .sortBy("y")
            .reverse()
            .take(10)
            .value();
        return (
            <div className="Histogram">
                <BarChart label={this.props.label} values={values} width={400} height={400} />
            </div>
        );
    }
}

export { BarChart, Histogram, Affinities };
