import React, { Component } from "react";
import XLSX from "xlsx";
import _ from "lodash";
import { Button, RadioButton } from "@lib/components/lego";
import Json from "@lib/components/json";
import FontAwesome from "react-fontawesome";

import Loading from "@lib/components/loading";
import SampleStatsTables from "./SampleStatsTables";
import SampleStatsCharts from "./SampleStatsCharts";
import { getAoaStats, getPrettyStats } from "./stats";
import { downloadPlainTextFile } from "@lib/utils/files";

import "./SampleStats.scss";
import { withTranslation } from "react-i18next";

@withTranslation("samples")
class SampleStats extends Component {
    constructor() {
        super();
        this.selectSampleStatsType = this.selectSampleStatsType.bind(this);
        this.state = { type: "picture", loadingPdf: false };
    }

    selectSampleStatsType(type) {
        this.setState({ type: type });
    }

    getExportFilename(exportFileBasename, extension) {
        return exportFileBasename.replace(/[\s.]/g, "_") + extension;
    }

    getJsonStatsString(stats) {
        return JSON.stringify(stats.value, null, "  ");
    }

    getTsvStatsString(prettyStats) {
        return _.join(
            _.map(
                getAoaStats(prettyStats, this.props.i18n.language),
                (arrayStats) =>
                    _.join(
                        _.map(arrayStats, (each) => _.join(each, "\t")),
                        "\n"
                    )
            ),
            "\n\n"
        );
    }

    loadStatsTsvFile(prettyStats, exportFilename) {
        downloadPlainTextFile(
            this.getTsvStatsString(prettyStats),
            exportFilename
        );
    }

    loadStatsJsonFile(stats, exportFilename) {
        downloadPlainTextFile(this.getJsonStatsString(stats), exportFilename);
    }

    loadStatsXlsxFile(prettyStats, exportFilename) {
        const aoaStats = _.concat(
            ..._.map(
                getAoaStats(prettyStats, this.props.i18n.language),
                (each) => _.concat(each, [[]])
            )
        );
        const worksheet = XLSX.utils.aoa_to_sheet(aoaStats);
        const workbook = XLSX.utils.book_new();
        XLSX.utils.book_append_sheet(workbook, worksheet, "");
        XLSX.writeFile(workbook, exportFilename);
    }

    renderLoadingButtons(stats, prettyStatsAll, exportFileBasename) {
        const { t } = this.props;

        if (this.state.type === "table") {
            return (
                <span className="loadingButtons">
                    <Button
                        size="s"
                        theme="raised"
                        onClick={() =>
                            this.loadStatsTsvFile(
                                prettyStatsAll,
                                this.getExportFilename(
                                    exportFileBasename,
                                    ".tsv"
                                )
                            )
                        }
                    >
                        <FontAwesome name="file-download" />
                        {t("loadStats") + " .tsv"}
                    </Button>{" "}
                    <Button
                        size="s"
                        theme="raised"
                        onClick={() =>
                            this.loadStatsXlsxFile(
                                prettyStatsAll,
                                this.getExportFilename(
                                    exportFileBasename,
                                    ".xlsx"
                                )
                            )
                        }
                    >
                        <FontAwesome name="file-download" />
                        {t("loadStats") + " .xlsx"}
                    </Button>
                </span>
            );
        } else if (this.state.type === "text") {
            return (
                <span className="loadingButtons">
                    <Button
                        size="s"
                        theme="raised"
                        onClick={() =>
                            this.loadStatsJsonFile(
                                stats,
                                this.getExportFilename(
                                    exportFileBasename,
                                    ".json"
                                )
                            )
                        }
                    >
                        <FontAwesome name="file-download" />
                        {t("loadStats") + " .json"}
                    </Button>
                </span>
            );
        } else {
            return <span />;
        }
    }

    renderBody(stats, prettyStatsShorted) {
        const lang = this.props.i18n.language;

        if (this.state.type === "picture") {
            return (
                <SampleStatsCharts
                    prettyStats={prettyStatsShorted}
                    lang={lang}
                />
            );
        } else if (this.state.type === "table") {
            return (
                <SampleStatsTables
                    prettyStats={prettyStatsShorted}
                    lang={lang}
                />
            );
        } else if (this.state.type === "text") {
            return <Json data={this.getJsonStatsString(stats)} />;
        } else {
            return <div />;
        }
    }

    renderStatsContent() {
        const { stats, exportFileBasename } = this.props;
        const lang = this.props.i18n.language;

        const prettyStatsShorted = getPrettyStats(stats, lang, false);
        const prettyStatsAll = getPrettyStats(stats, lang, true);

        return (
            <div className="SampleStats">
                <div className="SampleStats-header">
                    <RadioButton
                        size="s"
                        value={this.state.type}
                        view="default"
                        onChange={(event) =>
                            this.selectSampleStatsType(event.target.value)
                        }
                        options={[
                            {
                                value: "picture",
                                children: <FontAwesome name="image" />,
                            },
                            {
                                value: "table",
                                children: <FontAwesome name="table" />,
                            },
                            {
                                value: "text",
                                children: <FontAwesome name="code" />,
                            },
                        ]}
                    />
                    {this.renderLoadingButtons(
                        stats,
                        prettyStatsAll,
                        exportFileBasename
                    )}
                </div>
                <div className="SampleStats-body">
                    {this.renderBody(stats, prettyStatsShorted)}
                </div>
            </div>
        );
    }

    render() {
        const { loaded, withLoading } = this.props;

        return withLoading ? (
            <Loading loaded={loaded}>{this.renderStatsContent()}</Loading>
        ) : (
            this.renderStatsContent()
        );
    }
}

export default SampleStats;
