import React, { Component } from "react";
import { Doughnut, HorizontalBar } from "react-chartjs-2";
import _ from "lodash";
import { CHARTS_TITLES, CHARTS_LABELS } from "./stats";

import "./SampleStatsCharts.scss";
import localized from "./localized";

const CHARTS_BACKGROUND_COLORS = [
    "rgba(255, 99, 132, 0.7)",
    "rgba(54, 162, 235, 0.7)",
    "rgba(255, 206, 86, 0.7)",
    "rgba(255, 159, 64, 0.7)",
    "rgba(75, 192, 192, 0.7)",
    "rgba(149, 117, 205, 0.7)",
    "rgba(129, 199, 132, 0.7)",
    "rgba(255, 99, 132, 0.4)",
    "rgba(54, 162, 235, 0.4)",
    "rgba(255, 206, 86, 0.4)",
    "rgba(255, 159, 64, 0.4)",
    "rgba(75, 192, 192, 0.4)",
    "rgba(149, 117, 205, 0.4)",
    "rgba(129, 199, 132, 0.4)",
    "rgba(188, 170, 164, 0.4)",
];

const CHARTS_UNKNOWN_BACKGROUND_COLOR = "rgba(189, 189, 189, 0.3)";
const CHARTS_BACKGROUND_COLOR_IDS = {
    Host: 0,
    Word: 1,
    "Segments-Demographics+": 3,
    "Segments-Behaviors": 4,
    "Segments-Demographics+_Behaviors": 5,
    "Segments-Common": 4,
    "Segments-InterestsShortterm": 6,
    "Segments-InterestsLongterm": 3,
    "Segments-Organizations": 2,
};

class StatsDoughnut extends Component {
    render() {
        const { type, prettyStats: stats, lang, height, width } = this.props;

        return (
            <div className="Chart">
                <Doughnut
                    height={height}
                    width={width}
                    data={{
                        datasets: [
                            {
                                data: _.map(stats[type], "Count"),
                                backgroundColor: CHARTS_BACKGROUND_COLORS.slice(0, stats[type].length - 1).concat(
                                    CHARTS_UNKNOWN_BACKGROUND_COLOR
                                ),
                            },
                        ],
                        labels: _.map(_.filter(stats[type], "Count"), "Name"),
                    }}
                    options={{
                        animation: {
                            duration: 0,
                        },
                        rotation: -Math.PI,
                        layout: {
                            padding: {
                                left: 20,
                                right: 20,
                            },
                        },
                        legend: {
                            position: "left",
                            labels: {
                                boxWidth: 10,
                                fontSize: 10,
                                fontFamily: "'YS Text Web', 'Helvetica', 'Arial', sans-serif",
                            },
                        },
                        title: {
                            display: true,
                            text: localized(_.get(CHARTS_TITLES, type), lang),
                            fontSize: 15,
                            fontFamily: "'YS Text Web', 'Helvetica', 'Arial', sans-serif",
                        },
                        tooltips: {
                            displayColors: false,
                            callbacks: {
                                label: function (tooltipItem, data) {
                                    let dataset = data.datasets[tooltipItem.datasetIndex];
                                    let meta = dataset._meta[Object.keys(dataset._meta)[0]];
                                    let total = meta.total;
                                    let currentValue = dataset.data[tooltipItem.index];
                                    let percentage = ((100 * currentValue) / total).toFixed(2);
                                    return Intl.NumberFormat(lang).format(currentValue) + " (" + percentage + "%)";
                                },
                                title: function (tooltipItem, data) {
                                    return data.labels[tooltipItem[0].index];
                                },
                            },
                        },
                    }}
                />
            </div>
        );
    }
}

class StatsBar extends Component {
    render() {
        const { type, prettyStats: stats, lang, height, width, nda } = this.props;

        return (
            <div className="Chart">
                <HorizontalBar
                    height={height}
                    width={width}
                    data={{
                        datasets: [
                            {
                                label: localized(CHARTS_LABELS.Affinity, lang),
                                data: _.map(stats[type], (each) =>
                                    typeof each.Affinity === "number" ? each.Affinity.toFixed(2) : "unknown"
                                ),
                                backgroundColor: CHARTS_BACKGROUND_COLORS[_.get(CHARTS_BACKGROUND_COLOR_IDS, type, 0)],
                                xAxisID: "top-x-axis",
                            },
                            {
                                label: localized(CHARTS_LABELS.Ratio, lang),
                                data: _.map(stats[type], (each) => (100 * each.Ratio).toFixed(2)),
                                xAxisID: "bottom-x-axis",
                            },
                        ],
                        labels: _.map(stats[type], "Name"),
                    }}
                    options={{
                        animation: {
                            duration: 0,
                        },
                        layout: {
                            padding: {
                                left: 40,
                                right: 40,
                            },
                        },
                        title: {
                            display: true,
                            text: localized(_.get(CHARTS_TITLES, type), lang) + (nda ? " (NDA)" : ""),
                            fontSize: 15,
                            fontFamily: "'YS Text Web', 'Helvetica', 'Arial', sans-serif",
                        },
                        legend: {
                            position: "bottom",
                            labels: {
                                boxWidth: 10,
                                fontSize: 10,
                                fontFamily: "'YS Text Web', 'Helvetica', 'Arial', sans-serif",
                            },
                        },
                        scales: {
                            yAxes: [
                                {
                                    barThickness: 7,
                                    maxbBarThickness: 7,
                                },
                            ],
                            xAxes: [
                                {
                                    id: "top-x-axis",
                                    type: "linear",
                                    position: "top",
                                    ticks: {
                                        maxTicksLimit: 4,
                                        beginAtZero: true,
                                    },
                                },
                                {
                                    id: "bottom-x-axis",
                                    type: "linear",
                                    position: "bottom",
                                    ticks: {
                                        maxTicksLimit: 4,
                                        beginAtZero: true,
                                    },
                                },
                            ],
                        },
                        tooltips: {
                            displayColors: true,
                            callbacks: {
                                label: function (tooltipItem, data) {
                                    let dataset = data.datasets[tooltipItem.datasetIndex];
                                    let currentValue = dataset.data[tooltipItem.index];
                                    return " " + Intl.NumberFormat(lang).format(currentValue) + " " + dataset.label;
                                },
                                title: function (tooltipItem, data) {
                                    return data.labels[tooltipItem[0].index];
                                },
                            },
                        },
                    }}
                />
            </div>
        );
    }
}

class SampleStatsCharts extends Component {
    render() {
        const { prettyStats, lang } = this.props;

        if (prettyStats) {
            const total = _.sumBy(prettyStats.Gender, "Count");

            return (
                <div id="SampleStatsCharts" className="SampleStatsCharts">
                    <div key="title" className="SampleStatsCharts-info">
                        <p>
                            {localized(_.get(CHARTS_TITLES, "NumberOfIdsBegin"), lang) +
                                Intl.NumberFormat(lang).format(total) +
                                localized(_.get(CHARTS_TITLES, "NumberOfIdsEnd"), lang) + "."}
                        </p>
                    </div>
                    <div key="doughnuts" className="SampleStatsCharts-row">
                        {_.map(["Gender", "Age", "Income"], (each) => {
                            return (
                                <div className="SampleStatsCharts-socdem">
                                    <StatsDoughnut
                                        key={each}
                                        type={each}
                                        prettyStats={prettyStats}
                                        lang={lang}
                                        height={190}
                                        width={300}
                                    />
                                    <div className="StatsBar-socdem">
                                        <StatsBar
                                            key={each + "Affinity"}
                                            type={each + "Affinities"}
                                            prettyStats={prettyStats}
                                            lang={lang}
                                            height={270}
                                            width={360}
                                        />
                                    </div>
                                </div>
                            );
                        })}
                    </div>
                    <div key="device-region-doughnuts" className="SampleStatsCharts-row">
                        <StatsDoughnut
                            key="Device"
                            type="Device"
                            prettyStats={prettyStats}
                            lang={lang}
                            height={190}
                            width={300}
                        />
                        <StatsDoughnut
                            key="Region"
                            type="Region"
                            prettyStats={prettyStats}
                            lang={lang}
                            height={190}
                            width={600}
                        />
                    </div>
                    <div key="gender-age-income-doughnuts" className="SampleStatsCharts-row">
                        <StatsDoughnut
                            key="GenderAgeIncome"
                            type="GenderAgeIncome"
                            prettyStats={prettyStats}
                            lang={lang}
                            height={190}
                            width={900}
                        />
                    </div>
                    {_.map(
                        [
                            [{ chart: "Host", nda: true }, { chart: "Word", nda: false }],
                            [{ chart: "App", nda: true }],
                            [{ chart: "Segments-Demographics+_Behaviors", nda: false }, { chart: "Segments-Common", nda: false }],
                            [{ chart: "Segments-InterestsShortterm", nda: false }, { chart: "Segments-InterestsLongterm", nda: false }],
                            [{ chart: "Segments-Organizations", nda: false }],
                        ],
                        (eachRow) => {
                            return (
                                <div key={eachRow} className="SampleStatsCharts-row">
                                    {_.map(eachRow, (each) => (
                                        <StatsBar
                                            nda={each.nda}
                                            key={each.chart}
                                            type={each.chart}
                                            prettyStats={prettyStats}
                                            lang={lang}
                                            height={450}
                                            width={450}
                                        />
                                    ))}
                                </div>
                            );
                        }
                    )}
                </div>
            );
        } else {
            return null;
        }
    }
}

export default SampleStatsCharts;
