import React, { Component } from "react";
import _ from "lodash";
import localized from "./localized";
import { CHARTS_TITLES, CHARTS_LABELS } from "./stats";
import "./SampleStatsTables.scss";

class CountsTable extends Component {
    render() {
        const total = _.sumBy(this.props.stats[this.props.type], "Count");

        return (
            <table className="SampleStatsTable">
                <thead>
                    <tr className="SampleStatsTable-header">
                        <th>{localized(CHARTS_TITLES[this.props.type], this.props.lang)}</th>
                        <th>{localized(CHARTS_LABELS.Ratio, this.props.lang)}</th>
                    </tr>
                </thead>
                <tbody>
                    {
                        // TODO: find out why some each may be undefined
                        _.map(
                            this.props.stats[this.props.type].filter((item) => !_.isUndefined(item)),
                            (each) => {
                                return (
                                    <tr className="SampleStatsTable-row" key={each.Name}>
                                        <td>{each.Name}</td>
                                        <td>{((100 * each.Count) / total).toFixed(2)}</td>
                                    </tr>
                                );
                            }
                        )
                    }
                </tbody>
            </table>
        );
    }
}

class AffinitiesTable extends Component {
    render() {
        const typeShortened = this.props.type.split("-", 1)[0];

        const getLink = (item) => {
            switch (typeShortened) {
                case "Host":
                    return "https://" + item.Name;
                case "Word":
                    return "https://yandex.ru/search/?text=" + item.Name;
                case "Segments":
                    return `${
                        window.location.href.split("/", 1)[0]
                    }/segments?group=${
                        item.Parent[item.Parent.length - 2].ID
                    }&segment=${item.ID}`;
                default:
                    return "";
            }
        };

        return (
            <table className="SampleStatsTable">
                <thead>
                    <tr className="SampleStatsTable-header">
                        <th>{localized(CHARTS_TITLES[this.props.type], this.props.lang)}</th>
                        <th>{localized(CHARTS_LABELS.Affinity, this.props.lang)}</th>
                        <th>{localized(CHARTS_LABELS.Ratio, this.props.lang)}</th>
                    </tr>
                </thead>
                <tbody>
                    {_.map(this.props.stats[this.props.type], (each, i) => {
                        return (
                            <tr className="SampleStatsTable-row" key={i + each.Name}>
                                <td>
                                    <a href={getLink(each)} target="_blank">
                                        {each.Name}
                                    </a>
                                </td>
                                <td>{each.Affinity.toFixed(1)}</td>
                                <td>{(100 * each.Ratio).toFixed(2)}</td>
                            </tr>
                        );
                    })}
                </tbody>
            </table>
        );
    }
}

class SampleStatsTables extends Component {
    render() {
        const { prettyStats, lang } = this.props;

        if (prettyStats) {
            return (
                <div id="SampleStatsTables" className="SampleStatsTables">
                    <div>
                        <table className="SampleStatsTable">
                            <thead>
                                <tr className="SampleStatsTable-header">
                                    <th>{localized(_.get(CHARTS_TITLES, "NumberOfIdsBegin"), lang) + localized(_.get(CHARTS_TITLES, "NumberOfIdsEnd"), lang) + ":"}</th>
                                </tr>
                            </thead>
                            <tbody>
                                <tr className="SampleStatsTable-row">
                                    <td>{_.sumBy(prettyStats.Gender, "Count")}</td>
                                </tr>
                            </tbody>
                        </table>
                    </div>
                    <div>
                        <CountsTable type="Gender" stats={prettyStats} lang={lang} />
                        <CountsTable type="Age" stats={prettyStats} lang={lang} />
                        <CountsTable type="Income" stats={prettyStats} lang={lang} />
                    </div>
                    <div>
                        <CountsTable type="Device" stats={prettyStats} lang={lang} />
                        <CountsTable type="GenderAgeIncome" stats={prettyStats} lang={lang} />
                    </div>
                    <div>
                        <CountsTable type="Region" stats={prettyStats} lang={lang} />
                        <AffinitiesTable type="Host" stats={prettyStats} lang={lang} />
                        <AffinitiesTable type="Word" stats={prettyStats} lang={lang} />
                    </div>
                    <div>
                        <AffinitiesTable type="App" stats={prettyStats} lang={lang} />
                    </div>
                    <div>
                        <AffinitiesTable type="Segments-Demographics+_Behaviors" stats={prettyStats} lang={lang} />
                        <AffinitiesTable type="Segments-Common" stats={prettyStats} lang={lang} />
                    </div>
                    <div>
                        <AffinitiesTable type="Segments-InterestsShortterm" stats={prettyStats} lang={lang} />
                        <AffinitiesTable type="Segments-InterestsLongterm" stats={prettyStats} lang={lang} />
                    </div>
                    <div>
                        <AffinitiesTable type="Segments-Organizations" stats={prettyStats} lang={lang} />
                    </div>
                </div>
            );
        } else {
            return null;
        }
    }
}

export default SampleStatsTables;
