import _ from "lodash";
import punycode from "punycode";

import localized from "./localized";

export const CHARTS_LABELS = {
    Gender: {
        MALE: { ru: "мужской", en: "male", emoji: "👨" },
        FEMALE: { ru: "женский", en: "female", emoji: "👩" },
    },
    Age: {
        FROM_0_TO_17: { ru: "до 18", en: "under 18", emoji: "" },
        FROM_18_TO_24: { ru: "18–24", en: "18–24", emoji: "" },
        FROM_25_TO_34: { ru: "25–34", en: "25–34", emoji: "" },
        FROM_35_TO_44: { ru: "35–44", en: "35–44", emoji: "" },
        FROM_45_TO_54: { ru: "45–54", en: "45–54", emoji: "" },
        FROM_55_TO_99: { ru: "от 55", en: "over 55", emoji: "" },
    },
    Income: {
        INCOME_A: { ru: "низкий", en: "low", emoji: "" },
        INCOME_B1: { ru: "ниже среднего", en: "below average", emoji: "" },
        INCOME_B2: { ru: "выше среднего", en: "above average", emoji: "" },
        INCOME_C1: { ru: "высокий", en: "high", emoji: "" },
        INCOME_C2: { ru: "премиум", en: "premium", emoji: "" },
    },
    Device: {
        DESKTOP: { ru: "компьютер", en: "desktop", emoji: "" },
        PHONE: { ru: "телефон", en: "phone", emoji: "" },
        TABLET: { ru: "планшет", en: "tablet", emoji: "" },
        MIXED: { ru: "смешанный", en: "mixed", emoji: "" },
    },
    Affinity: { ru: "афинити", en: "affinity" },
    Ratio: { ru: "% пользователей", en: "% of users" },
};

const CHARTS_UNKNOWN_LABEL = { ru: "неизвестно", en: "unknown" };
const CHARTS_OTHERS_LABEL = { ru: "прочие", en: "others" };

export const CHARTS_TITLES = {
    Total: { ru: "Всего", en: "Total" },
    Gender: { ru: "Пол", en: "Gender" },
    Age: { ru: "Возраст", en: "Age" },
    Income: { ru: "Доход", en: "Income" },
    GenderAgeIncome: { ru: "Пол, Возраст, Доход", en: "Gender, Age, Income" },
    Device: { ru: "Устройство", en: "Device" },
    Region: { ru: "Город проживания", en: "City" },
    Host: { ru: "Сайты", en: "Hosts" },
    Word: { ru: "Слова", en: "Words" },
    App: { ru: "Приложения", en: "Apps" },
    GenderAffinities: { ru: "Афинити по полу", en: "Gender affinities" },
    AgeAffinities: { ru: "Афинити по возрасту", en: "Age affinities" },
    IncomeAffinities: { ru: "Афинити по доходу", en: "Income affinities" },
    NumberOfIdsBegin: { ru: "Для описания мы используем подвыборку из 100 тыс. идентификаторов. Для нее нашлось ",
        en: "We select random 100K ids for description. There are " },
    NumberOfIdsEnd: { ru: " активных crypta_id", en: " active crypta_id found for it" },
    "Segments-Demographics+": { ru: "Сегменты, Демография+", en: "Segments, Demographics+" },
    "Segments-Behaviors": { ru: "Сегменты, Поведение", en: "Segments, Behaviors" },
    "Segments-Demographics+_Behaviors": {
        ru: "Сегменты, Демография+ и Поведение",
        en: "Segments, Demographics+ & Behaviors",
    },
    "Segments-Common": { ru: "Сегменты, Категории общие", en: "Segments, Categories common" },
    "Segments-InterestsShortterm": {
        ru: "Сегменты, Интересы 2.0 краткосрочные",
        en: "Segments, Interests 2.0 shortterm",
    },
    "Segments-InterestsLongterm": { ru: "Сегменты, Интересы 2.0 долгосрочные", en: "Segments, Interests 2.0 longterm" },
    "Segments-Organizations": { ru: "Сегменты, Посещения организаций", en: "Segments, Organization visits" },
};

const NUMBER_OF_REGIONS = 14;
const NUMBER_OF_VALUES = 15;
const NUMBER_OF_GENDERAGEINCOME = 14;

function getCountsPrettyStats(type, stats, lang) {
    return _.map(_.sortBy(stats.value.Counts["By" + type], type), (each) => {
        return {
            Name: localized(_.get(CHARTS_LABELS[type], each[type], CHARTS_UNKNOWN_LABEL), lang),
            Count: each.Count,
        };
    });
}

function getGenderAgeIncomeStats(type, stats, lang, full) {
    const COUNT_OF_ROWS = full
        ? stats.value.Counts.ByGenderAgeIncome.length
        : _.min([NUMBER_OF_GENDERAGEINCOME, stats.value.Counts.ByGenderAgeIncome.length]);

    const genderageincomeStats = _.map(
        _.sortBy(stats.value.Counts.ByGenderAgeIncome, "Count").reverse().slice(0, COUNT_OF_ROWS),
        (each) => {
            return {
                Name: _.join(
                    _.map(["Gender", "Age", "Income"], (socdem_type) =>
                        localized(
                            _.get(CHARTS_LABELS[socdem_type], each.GenderAgeIncome[socdem_type], CHARTS_UNKNOWN_LABEL),
                            lang
                        )
                    ),
                    ", "
                ),
                Count: each.Count,
            };
        }
    );
    if (COUNT_OF_ROWS < stats.value.Counts.ByGenderAgeIncome.length) {
        genderageincomeStats[COUNT_OF_ROWS] = {
            Name: localized(CHARTS_OTHERS_LABEL, lang),
            Count: _.sumBy(
                _.sortBy(stats.value.Counts.ByGenderAgeIncome, "Count")
                    .reverse()
                    .slice(COUNT_OF_ROWS + 1),
                "Count"
            ),
        };
    }
    return genderageincomeStats;
}

function getRegionsPrettyStats(type, stats, lang) {
    const unknown = _.find(stats.value.Counts.ByRegion, ["ID", 0]);
    const sortedStats = _.sortBy(
        _.filter(stats.value.Counts.ByRegion, (each) => each.ID !== 0),
        "Count"
    ).reverse();
    const sortedStatsShortened = sortedStats.slice(0, NUMBER_OF_REGIONS);
    sortedStatsShortened[NUMBER_OF_REGIONS] = {
        Name: localized(CHARTS_OTHERS_LABEL, lang),
        Count: _.sumBy(sortedStats.slice(NUMBER_OF_REGIONS), "Count"),
    };
    sortedStatsShortened[NUMBER_OF_REGIONS + 1] = { ...unknown, Name: localized(CHARTS_UNKNOWN_LABEL, lang) };
    return sortedStatsShortened;
}

function getAffinitiesPrettyStats(type, stats) {
    const topValues = _.sortBy(stats.value.Affinities["By" + type], "Affinity")
        .reverse()
        .slice(0, NUMBER_OF_VALUES);
    return _.map(topValues, (each) => {
        return {
            Name: punycode.toUnicode(each.Value),
            Ratio: each.Ratio,
            Affinity: each.Affinity,
        };
    });
}

function getSegmentsPrettyStats(segmentGroups, exportType, stats) {
    const segmentsStatsFiltered = _.filter(stats.value.Affinities.BySegment, (each) => {
        return (
            segmentGroups.includes(_.slice(each.Parent, -3)[0].ID) &&
            (!exportType || each.ExportType === exportType) &&
            each.Affinity > 1
        );
    });

    return _.sortBy(segmentsStatsFiltered, "Affinity").reverse().slice(0, NUMBER_OF_VALUES);
}

function getSocdemPrettyStats(type, stats, lang) {
    const socdemStatsByType = stats.value.Affinities["By" + type];

    return _.sortBy(socdemStatsByType, "Affinity")
        .reverse()
        .map((each) => ({
            Name: localized(_.get(CHARTS_LABELS[type], each[type], CHARTS_UNKNOWN_LABEL), lang),
            Affinity: each.Affinity,
            Ratio: each.Ratio,
        }));
}

export function getPrettyStats(stats, lang, full = false) {
    if (stats && stats.value) {
        return {
            Gender: getCountsPrettyStats("Gender", stats, lang),
            Age: getCountsPrettyStats("Age", stats, lang),
            Income: getCountsPrettyStats("Income", stats, lang),
            GenderAgeIncome: getGenderAgeIncomeStats("GenderAgeIncome", stats, lang, full),
            Device: getCountsPrettyStats("Device", stats, lang),
            Region: getRegionsPrettyStats("Region", stats, lang),
            Host: getAffinitiesPrettyStats("Host", stats),
            Word: getAffinitiesPrettyStats("Word", stats),
            App: getAffinitiesPrettyStats("App", stats),
            GenderAffinities: getSocdemPrettyStats("Gender", stats, lang),
            AgeAffinities: getSocdemPrettyStats("Age", stats, lang),
            IncomeAffinities: getSocdemPrettyStats("Income", stats, lang),
            "Segments-Demographics+_Behaviors": getSegmentsPrettyStats(
                ["group-01f9d150", "group-9190c100"],
                "DEFAULT",
                stats
            ),
            "Segments-Common": getSegmentsPrettyStats(["group-62df3c1e"], "DEFAULT", stats),
            "Segments-InterestsShortterm": getSegmentsPrettyStats(["group-d0802a75"], "SHORTTERM", stats),
            "Segments-InterestsLongterm": getSegmentsPrettyStats(["group-d0802a75"], "LONGTERM", stats),
            "Segments-Organizations": getSegmentsPrettyStats(["group-8130912a"], "DEFAULT", stats),
        };
    }
}

function getTotalAoaStats(prettyStats, lang) {
    return [
        [localized(CHARTS_TITLES.Total, lang), ""],
        ["yandexuid", _.sumBy(prettyStats.Gender, "Count")],
    ];
}

function getCountsAoaStats(type, prettyStats, lang) {
    const total = _.sumBy(prettyStats[type], "Count");
    return [
        _.map([CHARTS_TITLES[type], CHARTS_LABELS.Ratio], (each) => localized(each, lang)),
        // FIXME: find out why each may be undefined
        ..._.map(
            prettyStats[type].filter((item) => !_.isUndefined(item)),
            (each) => [each.Name, ((100 * each.Count) / total).toFixed(4)]
        ),
    ];
}

function mask(value) {
    return value.substr(0, 1) + "***" + value.substr(-2, 1);
}

function getAffinitiesAoaStats(type, prettyStats, lang, shouldMask) {
    let masker = shouldMask ? mask : (value) => value;
    return [
        _.map([CHARTS_TITLES[type], CHARTS_LABELS.Affinity, CHARTS_LABELS.Ratio], (each) => localized(each, lang)),
        ..._.map(prettyStats[type], (each) => [
            masker(each.Name.replace("\t", " ")),
            each.Affinity.toFixed(4),
            (100 * each.Ratio).toFixed(4),
        ]),
    ];
}

export function getAoaStats(stats, lang) {
    return [
        getTotalAoaStats(stats, lang),
        ..._.map(["Gender", "Age", "Income", "Device", "Region", "GenderAgeIncome"], (each) =>
            getCountsAoaStats(each, stats, lang)
        ),
        ..._.map(["Host", "Word", "App"], (each) => getAffinitiesAoaStats(each, stats, lang, true)),
        ..._.map(
            [
                "Segments-Demographics+_Behaviors",
                "Segments-Common",
                "Segments-InterestsShortterm",
                "Segments-InterestsLongterm",
                "Segments-Organizations",
            ],
            (each) => getAffinitiesAoaStats(each, stats, lang, false)
        ),
    ]; // Array of arrays
}
