import os
import sys
import logging
import requests as r
import json
import socket

from blackboxer import Blackbox
from blackboxer.environment import ENV as BLACKBOX_ENVIRONMENT
import tvmauth

from flask import (
    Flask, request, redirect
)

from flask_cors import CORS


logger = logging.getLogger(__name__)
logger.setLevel(logging.DEBUG)
handler = logging.StreamHandler(sys.stdout)
handler.setLevel(logging.DEBUG)
formatter = logging.Formatter('%(asctime)s - %(name)s - %(levelname)s - %(message)s')
handler.setFormatter(formatter)
logger.addHandler(handler)


API_URL = os.environ.get('CRYPTA_API') or 'https://api-test.crypta.yandex-team.ru'
API_CHECK_PASSPORT_URL = '%s/lab/check_login' % API_URL
API_CONFIRM_AUTH_URL = '%s/lab/audience/grab/confirm_auth' % API_URL
API_OAUTH_TOKEN = os.environ.get('OAUTH_TOKEN')
APP = Flask('passport')
CORS(APP, resources={r"/passport/*": {"origins": "https://lab.crypta.yandex-team.ru"}})

BLACKBOX_ENV = os.environ.get('BLACKBOX_ENV') or 'test'
BLACKBOX_URL = BLACKBOX_ENVIRONMENT.get(BLACKBOX_ENV)
BLACKBOX = Blackbox(BLACKBOX_URL, 3, 2)
BLACKBOX_TVM_ID = int(os.environ.get('BLACKBOX_TVM_ID') or 224)

API_TVM_ID = int(os.environ.get('API_TVM_ID') or 2000761)
API_TVM_SECRET = os.environ.get('API_TVM_SECRET')
DEBUG_TICKET = os.environ.get('DEBUG_TICKET')

APP_IP_ADDRESS = None

APP_HOSTNAME = os.environ.get('APP_HOSTNAME') or ''
APP_PORT = int(os.environ.get('APP_PORT') or 9999)

USER_TICKET_CHECKING_ENDPOINT = tvmauth.BlackboxEnv.Test if BLACKBOX_ENV == 'test' else tvmauth.BlackboxEnv.Prod
PASSPORT_URL = 'passport-test.yandex.ru' if BLACKBOX_ENV == 'test' else 'passport.yandex.ru'


def tvm(self_secret):
    return tvmauth.TvmClient(tvmauth.TvmApiClientSettings(
        self_tvm_id=API_TVM_ID,
        self_secret=self_secret,
        enable_service_ticket_checking=True,
        enable_user_ticket_checking=USER_TICKET_CHECKING_ENDPOINT,
        dsts={'api': API_TVM_ID, 'blackbox': BLACKBOX_TVM_ID}
    ))


def call_blackbox(**kwargs):
    tvm_client = tvm(API_TVM_SECRET)

    if DEBUG_TICKET:
        service_ticket = DEBUG_TICKET
    else:
        service_ticket = tvm_client.get_service_ticket_for('blackbox')

    current_login = kwargs['current_login']
    grab_id = kwargs['grab_id']

    app_hostname = APP_HOSTNAME
    app_ip = APP_IP_ADDRESS
    session = request.cookies.get("Session_id")

    if not session:
        logger.error("Empty Session_id for login '%s'" % current_login)
        return open_auth(current_login, grab_id)

    blackbox = BLACKBOX.sessionid(
        app_ip,
        session,
        app_hostname,
        headers={'X-Ya-Service-Ticket': service_ticket}, **kwargs
    )

    if blackbox['status']['value'] == 'VALID':
        if current_login == blackbox['login']:
            confirm_auth(grab_id)

            return APP.response_class(
                response=json.dumps('OK, you are logged in'),
                mimetype='application/json',
                status=200
            )
        else:
            return open_auth(current_login, grab_id)
    else:
        logger.error("Blackbox status: " + str(blackbox))
        return open_auth(current_login, grab_id)


@APP.errorhandler(404)
def page_not_found(e):
    return APP.response_class(status=404)


@APP.errorhandler(500)
def internal_error(e):
    error_code = 500
    APP.logger.error(str(e))
    return APP.response_class(
        response="%d Internal server error" % error_code,
        status=error_code
    )


def open_auth(login, grab_id):
    retpath = 'https://' + APP_HOSTNAME + '/?login=' + login + "%26grab_id=" + grab_id
    backpath = retpath
    passport_url = 'https://' + PASSPORT_URL + '/auth?origin=crypta_lab&retpath=' + retpath + '&backpath=' + backpath

    return redirect(passport_url, code=302)


@APP.route('/', methods=["OPTIONS", "POST", "GET"])
def start():
    login = request.args.get('login')
    grab_id = request.args.get('grab_id')

    if login is None or grab_id is None:
        return "Bad request", 400

    # go to passport
    return call_blackbox(current_login=login, grab_id=grab_id)


def confirm_auth(grab_id):
    response = r.post(API_CONFIRM_AUTH_URL, data=json.dumps({}), verify=False,
                      params={'grab_id': grab_id},
                      headers={"Authorization": "OAuth %s" % API_OAUTH_TOKEN, 'Content-type': 'application/json'})

    if response.status_code not in [200, 204]:
        logger.error(str(response.status_code) + " Error confirming auth, grab_id=" + grab_id)


@APP.after_request
def patch_headers(response):
    response.headers['Server'] = 'nginx'
    response.headers['Content-Security-Policy'] = "default-src 'self';"
    response.headers['X-Frame-Options'] = "SAMEORIGIN"
    response.headers['X-XSS-Protection'] = "1; mode=block"
    response.headers['X-Content-Type-Options'] = "nosniff"
    response.headers['Access-Control-Allow-Origin'] = "https://lab.crypta.yandex-team.ru"
    response.headers['Access-Control-Allow-Credentials'] = 'true'
    response.headers['Access-Control-Allow-Methods'] = 'GET,PUT,POST,DELETE,OPTIONS'
    response.headers['Access-Control-Allow-Headers'] = 'Content-Type, *'
    return response


@APP.route('/favicon.ico')
def favicon():
    return APP.response_class(response='', status=200)


@APP.route('/health')
def health():
    return APP.response_class(response='OK', status=200)


def main():
    global APP_IP_ADDRESS
    if os.environ.get('QLOUD_IPV6'):
        APP_IP_ADDRESS = os.environ.get('QLOUD_IPV6').split('/')[0]
    elif os.environ.get('DEPLOY_POD_IP_0_ADDRESS'):
        APP_IP_ADDRESS = os.environ.get('DEPLOY_POD_IP_0_ADDRESS')
    else:
        APP_IP_ADDRESS = socket.gethostbyname(socket.gethostname())

    APP.logger.info('Starting APP')
    APP.run(host='::', port=int(os.environ.get('PORT', APP_PORT)), debug=False)
