import os
import logging
import requests as r
import json
from time import time

from flask import (
    Flask,
    request,
    render_template_string,
)

import library.python.resource as rs

logger = logging.getLogger(__name__)

API_URL = os.environ.get("CRYPTA_API")
API_GRAB_URL = '%s/portal/grab' % API_URL
OAUTH_TOKEN = os.environ.get('OAUTH_TOKEN')
APP = Flask('grab')


def get_yandexuid():
    return request.cookies.get('yandexuid', None)


def get_yandex_login():
    return request.cookies.get('yandex_login', None)


def get_useragent():
    return request.headers.get('User-Agent')


@APP.errorhandler(404)
def page_not_found(e):
    return APP.response_class(status=404)


@APP.errorhandler(500)
def internal_error(e):
    error_code = 500
    APP.logger.error(str(e))
    return APP.response_class(
        response="%d Internal server error" % error_code,
        status=error_code
    )


@APP.route('/<string:grab_id_encoded>')
def start(grab_id_encoded):

    yandexuid = get_yandexuid()
    yandex_login = get_yandex_login() or ""
    useragent = get_useragent()
    timestamp = int(time())

    if not yandexuid:
        APP.logger.info('User has no cookie')
        return APP.response_class(
            response=render_template_string(rs.find('/no_cookie_template.html').decode('utf-8'),
                                            status=200)
        )

    APP.logger.info("Found cookie: %s, login: %s" % (yandexuid, yandex_login))

    response = api_request(data_payload={"grabId": grab_id_encoded, "login": yandex_login,
                                         "useragent": useragent, "yandexuid": yandexuid,
                                         "timestamp": timestamp}
                           )

    APP.logger.info(response._content)

    return APP.response_class(response='OK', status=200)


def api_request(data_payload):
    url = API_GRAB_URL
    response = r.post(url, data=json.dumps(data_payload), verify=False,
                      headers={"Authorization": "OAuth %s" % OAUTH_TOKEN,
                               'Content-type': 'application/json'})
    return response


@APP.after_request
def patch_headers(response):
    host = request.headers['Host']
    login = request.cookies.get('yandex_login') or ''
    yandexuid = request.cookies.get('yandexuid') or ''

    response.headers['Server'] = 'nginx'
    response.headers['Content-Security-Policy'] = "default-src 'none';" + \
                                                  "img-src 'self';" + \
                                                  "report-uri https://csp.yandex.net/csp?from=" + host + \
                                                  "&project=crypta&yandex_login=" + login + "&yandexuid=%s;" % yandexuid
    response.headers['X-Frame-Options'] = "SAMEORIGIN"
    response.headers['X-XSS-Protection'] = "1; mode=block"
    response.headers['X-Content-Type-Options'] = "nosniff"
    return response


@APP.route('/favicon.ico')
def favicon():
    return APP.response_class(response='', status=200)


@APP.route('/health')
def health():
    return APP.response_class(response='OK', status=200)


def main():
    APP.logger.info('Starting APP')
    APP.run(host='::', port=int(os.environ.get('PORT', 80)), debug=False)
