import React, { Component } from "react";
import { isEmpty, isUndefined } from "lodash";
import { connect } from "react-redux";
import { Tab, TabList, TabPanel, Tabs } from "react-tabs";

import Panel from "@lib/components/panel";
import Json from "@lib/components/json";
import Link from "@lib/components/link";
import PCode from "@lib/components/pcode";
import OptionalTime from "@lib/components/time";
import { downloadPlainTextFile } from "@lib/utils/files";
import { Button } from "@lib/components/lego";

import FontAwesome from "react-fontawesome";
import OptionalQueryId from "./OptionalQueryId";

import LoadingSpinner from "../common/LoadingSpinner";
import ActionButton from "../common/ActionButton";
import ModalClosable from "../common/ModalClosable";
import { fetchSearchText, fetchSearchTextMatchedIfNeeded } from "src/ad/searchText/actions";
import { getAdParameters } from "src/selectors/ad";
import { getUid } from "src/selectors/uid";
import { fetchSelectTypes } from "./selectTypes/actions";

function tryParse(json) {
    try {
        return JSON.parse(json);
    } catch {
        return json;
    }
}

function loadingAd() {
    return (
        <div className="loading-ad">
            <div style={{ margin: "auto" }}>
                <LoadingSpinner size="l" />
                {" Loading ad..."}
            </div>
        </div>
    );
}

@connect(
    (state) => ({
        selectTypes: state.selectTypes.items,
        ad: state.ad,
        params: getAdParameters(state),
        searchText: state.searchText,
        uid: getUid(state),
    }),
    (dispatch) => ({
        fetchSelectTypes: () => dispatch(fetchSelectTypes()),
        fetchSearchText: (uidType, uid) => dispatch(fetchSearchText(uidType, uid)),
        fetchSearchTextMatchedIfNeeded: (uidType, uid) => dispatch(fetchSearchTextMatchedIfNeeded(uidType, uid)),
        dispatch: dispatch,
    })
)
export default class AdResponseTabs extends Component {
    componentDidMount() {
        this.props.fetchSelectTypes();
        this.props.fetchSearchText(this.props.uidType, this.props.uid);
    }

    componentWillReceiveProps(nextProps) {
        this.props.fetchSearchTextMatchedIfNeeded(this.props.uidType, nextProps.uid);
    }

    renderAd(ad, searchText, index) {
        let adv, adDebug, debugInfo;
        let { debugMode } = this.props.params;

        const properMode = debugMode === "ADVANCED" || debugMode === "MX";

        if (debugMode === "SIMPLE") {
            adDebug = ad.debug.replace(/\\"/g, '"').replace(/<br>/g, "");

            debugInfo =
                typeof ad.debug !== "undefined" && ad.debug !== "" ? (
                    <td className="col-md-4 banner-debug-cell">
                        <div className="banner-debug-content" dangerouslySetInnerHTML={{ __html: adDebug }}></div>
                    </td>
                ) : (
                    <td>{""}</td>
                );
        } else if (properMode && !isEmpty(this.props.ad.payload.adDebug)) {
            adDebug = this.props.ad.payload.adDebug[parseInt(ad.adId, 10)];

            let adSearchText;
            let queryIdHex;
            adv = <td />;

            if (!isEmpty(adDebug) && !isUndefined(adDebug)) {
                adSearchText = searchText[adDebug.QueryIDHex];
                queryIdHex =
                    adDebug.QueryIDHex.length < 16 && adDebug.QueryIDHex !== "0"
                        ? ("0000000000000000" + adDebug.QueryIDHex).substr(-16)
                        : adDebug.QueryIDHex;
            }

            if (this.props.searchText.isFetching) {
                adv = (
                    <td className="loading-search-text-cell">
                        <div>
                            <LoadingSpinner /> Checking for search_text for this ad...
                        </div>
                    </td>
                );
            } else if (!isUndefined(adSearchText) && !isEmpty(adSearchText)) {
                adv = (
                    <td className="col-md-4 search-text-debug-cell">
                        <table className="ad-debug-table">
                            <tbody>
                                <tr>
                                    <td className="ad-debug-title">create_time</td>
                                    <td>
                                        <OptionalTime value={parseInt(adSearchText["create_time"], 10)} />
                                    </td>
                                </tr>
                                <tr>
                                    <td className="ad-debug-title">parent_domain</td>
                                    <td>
                                        <Link href={"http://" + adSearchText["parent_domain"]} target="_blank">
                                            {adSearchText["parent_domain"]}
                                        </Link>
                                    </td>
                                </tr>
                                <tr>
                                    <td className="ad-debug-title">value</td>
                                    <td className="ad-debug-value">{adSearchText["value"]}</td>
                                </tr>
                                <tr>
                                    <td className="ad-debug-title">source uid</td>
                                    <td className="ad-debug-value">
                                        <Link
                                            href={`/profiles?uid=${adSearchText["uniq_id"]}&matching=0#query_id${queryIdHex}`}
                                            target="_blank"
                                        >
                                            {adSearchText["uniq_id"] + "  "}
                                            <FontAwesome name="external-link" />
                                        </Link>
                                    </td>
                                </tr>
                            </tbody>
                        </table>
                    </td>
                );
            } else {
                adv = <td />;
            }

            if (adDebug) {
                debugInfo = (
                    <td className="col-md-4 banner-debug-cell">
                        <table className="ad-debug-table">
                            <tbody>
                                <tr>
                                    <td className="ad-debug-title">SelectType</td>
                                    <td>
                                        {adDebug.SelectType}{" "}
                                        {this.props.selectTypes[adDebug.SelectType]
                                            ? this.props.selectTypes[adDebug.SelectType].description
                                            : ""}
                                    </td>
                                </tr>
                                <tr>
                                    <td className="ad-debug-title">BannerID</td>
                                    <td>{adDebug.BannerID}</td>
                                </tr>
                                <tr>
                                    <td className="ad-debug-title">CTR</td>
                                    <td>{adDebug.CTR}</td>
                                </tr>
                                <tr>
                                    <td className="ad-debug-title">BidCorrection</td>
                                    <td>{adDebug.BidCorrection}</td>
                                </tr>
                                <tr>
                                    <td className="ad-debug-title">Cost</td>
                                    <td>{adDebug.Cost}</td>
                                </tr>
                                <tr>
                                    <td className="ad-debug-title">PhraseID</td>
                                    <td>{adDebug.PhraseID}</td>
                                </tr>
                                <tr>
                                    <td className="ad-debug-title">PhrasePlusWords</td>
                                    <td>{adDebug.PhrasePlusWords}</td>
                                </tr>
                                <tr>
                                    <td className="ad-debug-title">QueryID</td>
                                    <td>
                                        <OptionalQueryId intView={adDebug.QueryID} hexView={queryIdHex} />
                                    </td>
                                </tr>
                                <tr>
                                    <td colSpan="2" className="ad-debug-title">
                                        {adSearchText ? (
                                            "search_text profile available"
                                        ) : (
                                            <div className="no-search-text">this ad has no search_text binding</div>
                                        )}
                                    </td>
                                </tr>
                            </tbody>
                        </table>
                    </td>
                );
            }
        }

        return (
            <Panel className="banner" key={index}>
                <table className="banner-table">
                    <tbody>
                        <tr>
                            <td className="banner-content-cell">
                                <Link href={ad.url} target="_blank">
                                    { ad.title }
                                </Link>
                                <div>{ ad.body }</div>
                                <Link href={"http://" + ad.domain} className="domain" target="_blank">
                                    {ad.domain}
                                </Link>
                                <div className="region">{ad.region}</div>
                            </td>
                            {adv}
                            {debugInfo}
                        </tr>
                    </tbody>
                </table>
            </Panel>
        );
    }

    renderAdsTab() {
        const searchText = this.props.searchText.value;
        const jsonBody = this.props.ad.payload.jsonBody;
        let ads;

        let { debugMode } = this.props.params;

        if (debugMode === "SIMPLE" && !isUndefined(jsonBody.result) && !isUndefined(jsonBody.result.ads)) {
            ads = jsonBody.result.ads;
        } else {
            ads = jsonBody.valid ? jsonBody.result.ads : [];
        }

        let tabContent;

        if (!isEmpty(ads)) {
            tabContent = ads.map((ad, index) => {
                return this.renderAd(ad, searchText, index);
            });
        } else {
            tabContent = "No such banners or got unsupported format";
        }

        return (
            <div>
                {tabContent}
            </div>
        );
    }

    render() {
        let ad = this.props.ad;
        let pcodeTabContent, bannersTabContent, jsonTabContent, rawTabContent, detailTabContent;

        let matrixnetFactors = null;
        let isMx = (this.props.params.debugMode == "MX");

        if (ad.isLoading) {
            pcodeTabContent = bannersTabContent = jsonTabContent = rawTabContent = detailTabContent = loadingAd();
        } else if (isEmpty(ad.payload)) {
            const empty = "empty";
            pcodeTabContent = empty;
            bannersTabContent = empty;
            jsonTabContent = empty;
            rawTabContent = empty;
            detailTabContent = empty;
        } else {
            // TODO: process if ad is fetched empty
            bannersTabContent = this.renderAdsTab();

            pcodeTabContent = (
                <Panel>
                    <PCode data={ad.payload.body} />
                </Panel>
            );

            jsonTabContent = (
                <Panel>
                    <Json data={tryParse(ad.payload.body)} />
                </Panel>
            );

            rawTabContent = (
                <Panel>
                    <Json data={ad.payload.body} />
                </Panel>
            );

            const escapedUrl = ad.payload.url.replace("[", "%5B").replace("]", "%5D");

            detailTabContent = (
                <Panel>
                    <div className="Details">
                        <h4 className="requestDetails">Cookie:</h4>
                        <pre>{ad.payload.headers.cookie}</pre>
                        <h4 className="requestDetails">User-Agent:</h4>
                        <pre>{ad.payload.headers["user-agent"]}</pre>
                        <h4 className="requestDetails">Raw url:</h4>
                        <p>
                            <Link rel="noreferrer" href={ad.payload.url}>
                                {ad.payload.url}
                            </Link>
                        </p>
                        <h4 className="requestDetails">Curl:</h4>
                        <pre>
                            curl -k -X GET -H "Cookie: {ad.payload.headers.cookie}" -H "User-Agent:{" "}
                            {ad.payload.headers["user-agent"]}" "{escapedUrl}"
                        </pre>
                    </div>
                </Panel>
            );

            if (isMx) {
                matrixnetFactors = JSON.stringify(ad.payload.mx, undefined, 2);
            }
        }

        return (
            <div>
                <Tabs>
                    <TabList>
                        <Tab key="pcode">PCode</Tab>
                        <Tab key="json">JSON</Tab>
                        <Tab key="banners">Banners</Tab>
                        <Tab key="raw">Raw</Tab>
                        <Tab key="detail">Details</Tab>
                        {
                            isMx &&
                                <Button
                                    disabled={ matrixnetFactors == null }
                                    view="raised"
                                    size="s"
                                    baseline="true"
                                    onClick={ () => downloadPlainTextFile(matrixnetFactors, "mx.json") }
                                >
                                    <FontAwesome name="download" />MX
                                </Button>
                        }
                    </TabList>
                    <TabPanel key="pcode">{pcodeTabContent}</TabPanel>
                    <TabPanel key="json">{jsonTabContent}</TabPanel>
                    <TabPanel key="banners">{bannersTabContent}</TabPanel>
                    <TabPanel key="raw">{rawTabContent}</TabPanel>
                    <TabPanel key="detail">{detailTabContent}</TabPanel>
                </Tabs>
            </div>
        );
    }
}
