import React, { Component } from "react";
import Autosuggest from "react-autosuggest";
import { connect } from "react-redux";
import { fetchUserAgentsIfNeeded } from "src/ad/useragent/actions";

import "./UserAgentInput.scss";

@connect(
    (state) => ({
        userAgents: state.userAgents,
    }),
    (dispatch) => ({
        fetchUserAgentsIfNeeded: () => dispatch(fetchUserAgentsIfNeeded()),
        dispatch: dispatch,
    })
)
export default class UserAgentInput extends Component {
    constructor() {
        super();
        this.onChange = this.onChange.bind(this);
        this.onSuggestionsFetchRequested = this.onSuggestionsFetchRequested.bind(this);
        this.onSuggestionsClearRequested = this.onSuggestionsClearRequested.bind(this);
        this.renderInputComponent = this.renderInputComponent.bind(this);
        this.onSuggestionSelected = this.onSuggestionSelected.bind(this);
        this.onEnterKeyPressed = this.onEnterKeyPressed.bind(this);

        this.state = {
            value: "",
            suggestions: [],
        };
    }

    componentDidMount() {
        this.props.fetchUserAgentsIfNeeded();
    }

    getSuggestions(event, userAgents) {
        let suggestions = [];
        if (userAgents) {
            for (let key in userAgents) {
                suggestions.push({ title: key, userAgents: userAgents[key] });
            }
        }

        return suggestions
            .map((section) => {
                return {
                    title: section.title,
                    userAgents: section.userAgents.filter((useragent) => {
                        return useragent.toUpperCase().includes(event.value.toUpperCase());
                    }),
                };
            })
            .filter((section) => section.userAgents.length > 0);
    }

    renderSuggestion(suggestion) {
        return <span>{suggestion}</span>;
    }

    renderSectionTitle(section) {
        return <strong>{section.title}</strong>;
    }

    renderInputComponent(inputProps) {
        return (
            <div>
                <input {...inputProps} />
                <span className="useragent-clear-button" onClick={() => this.props.onValueChange("")}>
                    ×
                </span>
            </div>
        );
    }

    onChange = (event, { newValue, method }) => {
        this.props.onValueChange(newValue);
    };

    onSuggestionSelected(event, { suggestion, suggestionValue, suggestionIndex, sectionIndex, method }) {
        this.props.onValueSelect(suggestionValue);
    }

    onEnterKeyPressed(event) {
        if (event.key === "Enter") {
            this.props.onValueSelect(event.target.value);
        }
    }

    onSuggestionsFetchRequested(value, userAgents) {
        this.setState({
            suggestions: this.getSuggestions(value, userAgents),
        });
    }

    onSuggestionsClearRequested = () => {
        this.setState({ suggestions: [] });
    };

    render() {
        const { userAgents } = this.props;
        const suggestions = this.state.suggestions;
        const inputProps = {
            placeholder: "Type useragent",
            value: this.props.value,
            onChange: this.onChange,
            onKeyPress: this.onEnterKeyPressed,
        };

        return (
            <Autosuggest
                multiSection={true}
                suggestions={suggestions}
                onSuggestionsFetchRequested={(value) => this.onSuggestionsFetchRequested(value, userAgents.items)}
                onSuggestionsClearRequested={this.onSuggestionsClearRequested}
                onSuggestionSelected={this.onSuggestionSelected}
                shouldRenderSuggestions={() => {
                    return true;
                }}
                getSuggestionValue={(suggestion) => {
                    return suggestion;
                }}
                renderSuggestion={this.renderSuggestion}
                renderSectionTitle={this.renderSectionTitle}
                getSectionSuggestions={(section) => {
                    return section.userAgents;
                }}
                renderInputComponent={this.renderInputComponent}
                inputProps={inputProps}
            />
        );
    }
}
