import React, { Component } from "react";
import Table from "@lib/components/table";
import _ from "lodash";

import OptionalTime from "@lib/components/time";
import { sortByInt, sortAB } from "../utils/sorters";
import CountersTable from "./CountersTable";
import Panel from "@lib/components/panel";

import "./CategoriesProfile.scss";

const PROFILE_NAME_CATEGORIES = "user-category-profile";
const PROFILE_NAME_COUNTERS = "bt-counter";
const NO_CATEGORIES_PROFILE = <div className="empty-content">{`User has no ${PROFILE_NAME_CATEGORIES} profile`}</div>;
const NO_COUNTERS_PROFILE = <div className="empty-content">{`User has no ${PROFILE_NAME_COUNTERS} profile`}</div>;

const FIELD_BM_CATEGORY_ID = "bm_category_id";
const FIELD_BM_CATEGORY = "bm_category";
const FIELD_EVENT_TIME = "event_time";
const FIELD_INTEREST_UPDATE_TIME = "interest_update_time";
const FIELD_INTEREST = "interest";
const FIELD_SHOWS = "shows";
const FIELD_CLICKS = "clicks";

const COUNTER_IDS = _.range(45, 55, 1).map((item) => item.toString());

const FIELD_CT_ID = "counter_id";
const FIELD_CT_CLICKS = "clicks";
const FIELD_CT_CLICK_TIME = "click_time";
const FIELD_CT_SHOWS = "shows";
const FIELD_CT_SHOW_TIME = "show_time";
const FIELD_CT_PREMIUM_CLICKS = "premium_clicks";
const FIELD_CT_PREMIUM_CLICK_TIME = "premium_click_time";
const FIELD_CT_MOUSEOVERS = "mouseovers";
const FIELD_CT_MOUSEOVER_TIME = "mouseover_time";
const FIELD_CT_ABUSES = "abuses";
const FIELD_CT_ABUSE_TIME = "abuse_time";

const COUNTER_ID_NAME_MAP = {
    "45": FIELD_CT_CLICKS,
    "46": FIELD_CT_CLICK_TIME,
    "47": FIELD_CT_SHOWS,
    "48": FIELD_CT_SHOW_TIME,
    "49": FIELD_CT_PREMIUM_CLICKS,
    "50": FIELD_CT_PREMIUM_CLICK_TIME,
    "51": FIELD_CT_MOUSEOVERS,
    "52": FIELD_CT_MOUSEOVER_TIME,
    "53": FIELD_CT_ABUSES,
    "54": FIELD_CT_ABUSE_TIME,
};

class CategoriesTable extends Component {
    constructor() {
        super();
        this.handleColumnNameClick = this.handleColumnNameClick.bind(this);

        this.state = {
            sortBy: FIELD_EVENT_TIME,
            descending: {},
        };
    }

    componentDidMount() {
        var descending = {};
        [FIELD_BM_CATEGORY_ID, FIELD_BM_CATEGORY, FIELD_INTEREST].forEach((item) => {
            descending[item] = true;
        });

        [FIELD_EVENT_TIME, FIELD_CLICKS, FIELD_SHOWS, FIELD_INTEREST_UPDATE_TIME].forEach((item) => {
            descending[item] = true;
        });

        this.setState({ descending: descending });
    }

    handleColumnNameClick(columnName) {
        var descending = Object.assign({}, this.state.descending);
        descending[columnName] =
            this.state.sortBy === columnName ? !this.state.descending[columnName] : this.state.descending[columnName];
        this.setState({
            sortBy: columnName,
            descending: descending,
        });
    }

    render() {
        const unsortedProfile = this.props.profile;

        const sorter = {
            [FIELD_BM_CATEGORY_ID]: (what) => {
                return sortByInt(FIELD_BM_CATEGORY_ID, what, this.state.descending[FIELD_BM_CATEGORY_ID]);
            },
            [FIELD_BM_CATEGORY]: (what) => {
                return sortAB(FIELD_BM_CATEGORY, what, this.state.descending[FIELD_BM_CATEGORY]);
            },
            [FIELD_EVENT_TIME]: (what) => {
                return sortByInt(FIELD_EVENT_TIME, what, this.state.descending[FIELD_EVENT_TIME]);
            },
            [FIELD_INTEREST_UPDATE_TIME]: (what) => {
                return sortByInt(FIELD_INTEREST_UPDATE_TIME, what, this.state.descending[FIELD_INTEREST_UPDATE_TIME]);
            },
            [FIELD_INTEREST]: (what) => {
                return sortByInt(FIELD_INTEREST, what, this.state.descending[FIELD_INTEREST]);
            },
            [FIELD_SHOWS]: (what) => {
                return sortByInt(FIELD_SHOWS, what, this.state.descending[FIELD_SHOWS]);
            },
            [FIELD_CLICKS]: (what) => {
                return sortByInt(FIELD_SHOWS, what, this.state.descending[FIELD_CLICKS]);
            },
        };

        const profile = sorter[this.state.sortBy](unsortedProfile);

        return (
            <div>
                <h4>User BM Categories</h4>
                <Table>
                    <thead>
                        <tr>
                            {[
                                FIELD_BM_CATEGORY_ID,
                                FIELD_BM_CATEGORY,
                                FIELD_EVENT_TIME,
                                FIELD_INTEREST_UPDATE_TIME,
                                FIELD_INTEREST,
                                FIELD_SHOWS,
                                FIELD_CLICKS,
                            ].map((columnName, key) => {
                                return (
                                    <th key={key}>
                                        <div
                                            className="categories-column-title"
                                            onClick={() => this.handleColumnNameClick(columnName)}
                                        >
                                            {columnName}
                                        </div>
                                    </th>
                                );
                            })}
                        </tr>
                    </thead>
                    <tbody>
                        {profile.map((item, key) => {
                            return (
                                <tr key={key}>
                                    {[
                                        FIELD_BM_CATEGORY_ID,
                                        FIELD_BM_CATEGORY,
                                        FIELD_EVENT_TIME,
                                        FIELD_INTEREST_UPDATE_TIME,
                                        FIELD_INTEREST,
                                        FIELD_SHOWS,
                                        FIELD_CLICKS,
                                    ].map((columnName, key) => {
                                        var value = "";
                                        if (item[columnName] !== null && typeof item[columnName] !== "undefined") {
                                            value = item[columnName];
                                            if ([FIELD_EVENT_TIME, FIELD_INTEREST_UPDATE_TIME].includes(columnName)) {
                                                value = <OptionalTime value={parseInt(value, 10)} />;
                                            }
                                        }

                                        return <td key={key}>{value}</td>;
                                    })}
                                </tr>
                            );
                        })}
                    </tbody>
                </Table>
            </div>
        );
    }
}

class CategoriesProfile extends Component {
    transposeProfile(profile) {
        var categories = {};

        // Further use specific counters only; filter by id
        var filteredCounters = profile.filter((item) => {
            return COUNTER_IDS.includes(item[FIELD_CT_ID]);
        });

        filteredCounters.forEach((counter) => {
            var categoryDict = counter["counters"];
            if (typeof categoryDict !== "undefined") {
                Object.keys(categoryDict).forEach((categoryId) => {
                    if (!Object.keys(categories).includes(categoryId)) {
                        categories[categoryId] = {};
                    }

                    var info = categoryDict[categoryId];
                    categories[categoryId][COUNTER_ID_NAME_MAP[counter[FIELD_CT_ID]]] = info["weight"];
                    categories[categoryId][FIELD_BM_CATEGORY_ID] = categoryId;
                });
            }
        });
        return _.values(categories);
    }

    renderProfile() {
        const categories = this.props.profile["user-category-profile"];
        const counters = this.transposeProfile(this.props.profile["bt-counter"]);

        var categoriesProfile =
            typeof categories === "undefined" ? NO_CATEGORIES_PROFILE : <CategoriesTable profile={categories} />;
        var countersProfile =
            typeof counters === "undefined" ? (
                NO_COUNTERS_PROFILE
            ) : (
                <CountersTable profile={counters} uid={this.props.uid} />
            );

        return (
            <div>
                {categoriesProfile}
                {countersProfile}
            </div>
        );
    }

    render() {
        const condition = typeof this.props.profile === "undefined";
        return (
            <Panel>
                {condition ? (
                    <div className="empty-content">{`User has neither ${PROFILE_NAME_CATEGORIES} nor ${PROFILE_NAME_COUNTERS}`}</div>
                ) : (
                    this.renderProfile()
                )}
            </Panel>
        );
    }
}

export default CategoriesProfile;
