import React, { Component } from "react";
import Table from "@lib/components/table";
import { isEmpty, isUndefined } from "lodash";

import OptionalTime from "@lib/components/time";
import { sortByInt, sortAB } from "../utils/sorters";
import Panel from "@lib/components/panel";
import Link from "@lib/components/link";

import "./EcomProfile.scss";

const PROFILE_NAME = "offer-profile";
const NO_PROFILE = <div className="empty-content">User has no ECOM profile</div>;

const FIELD_OFFER_NAME = "offer_name";
const FIELD_OFFER_TARGET_URL = "offer_target_url";
const FIELD_SITE_PATH = "site_path";
const FIELD_UPDATE_TIME = "update_time";
const FIELD_OFFER_ID_MD5 = "offer_id_md5";
const FIELD_COUNTER = "counter";
const FIELD_ACTION_BITS = "action_bits";
const FIELD_SELECT_TYPE = "select_type";
const FIELD_BID_MULTIPLIER = "bid_multiplier";
const FIELD_COUNTER_ID = "counter_id";
const FIELD_SOURCE_UNIQ_INDEX = "source_uniq_index";

const CUSTOM_FIELD_SOURCE_UID = "source uid";

class EcomTable extends Component {
    constructor() {
        super();
        this.handleColumnNameClick = this.handleColumnNameClick.bind(this);

        this.state = {
            sortBy: FIELD_UPDATE_TIME,
            descending: {},
        };
    }

    componentDidMount() {
        let descending = {};
        [
            FIELD_UPDATE_TIME,
            FIELD_OFFER_NAME,
            FIELD_SITE_PATH,
            FIELD_ACTION_BITS,
            FIELD_SELECT_TYPE,
            FIELD_BID_MULTIPLIER,
            FIELD_COUNTER_ID,
        ].forEach((item) => {
            descending[item] = true;
        });

        [FIELD_OFFER_ID_MD5, CUSTOM_FIELD_SOURCE_UID].forEach((item) => {
            descending[item] = false;
        });

        this.setState({ descending: descending });
    }

    handleColumnNameClick(columnName) {
        let descending = Object.assign({}, this.state.descending);
        descending[columnName] =
            this.state.sortBy === columnName ? !this.state.descending[columnName] : this.state.descending[columnName];
        this.setState({
            sortBy: columnName,
            descending: descending,
        });
    }

    render() {
        const sorter = {
            [FIELD_UPDATE_TIME]: (by, what) => {
                return sortByInt(by, what, this.state.descending[FIELD_UPDATE_TIME]);
            },
            [FIELD_OFFER_NAME]: (by, what) => {
                return sortAB(by, what, this.state.descending[FIELD_OFFER_NAME]);
            },
            [FIELD_SITE_PATH]: (by, what) => {
                return sortAB(by, what, this.state.descending[FIELD_SITE_PATH]);
            },
            [FIELD_OFFER_ID_MD5]: (by, what) => {
                return sortAB(by, what, this.state.descending[FIELD_OFFER_ID_MD5]);
            },
            [CUSTOM_FIELD_SOURCE_UID]: (by, what) => {
                return sortAB(by, what, this.state.descending[CUSTOM_FIELD_SOURCE_UID]);
            },
            [FIELD_ACTION_BITS]: (by, what) => {
                return sortAB(by, what, this.state.descending[FIELD_ACTION_BITS]);
            },
            [FIELD_SELECT_TYPE]: (by, what) => {
                return sortAB(by, what, this.state.descending[FIELD_SELECT_TYPE]);
            },
            [FIELD_BID_MULTIPLIER]: (by, what) => {
                return sortAB(by, what, this.state.descending[FIELD_BID_MULTIPLIER]);
            },
            [FIELD_COUNTER_ID]: (by, what) => {
                return sortAB(by, what, this.state.descending[FIELD_COUNTER_ID]);
            },
        };

        let compactProfile = [];

        this.props.profile.forEach((item) => {
            const sourceUniq = this.props.sourceUniqs[item[FIELD_SOURCE_UNIQ_INDEX]].user_id;

            compactProfile.push({
                [FIELD_UPDATE_TIME]: item[FIELD_UPDATE_TIME],
                [FIELD_OFFER_NAME]:
                    isUndefined(item[FIELD_OFFER_NAME]) || isEmpty(item[FIELD_OFFER_NAME])
                        ? ""
                        : item[FIELD_OFFER_NAME],
                [FIELD_OFFER_TARGET_URL]:
                    isUndefined(item[FIELD_OFFER_TARGET_URL]) || isEmpty(item[FIELD_OFFER_TARGET_URL])
                        ? ""
                        : item[FIELD_OFFER_TARGET_URL],
                [FIELD_SITE_PATH]:
                    isUndefined(item[FIELD_COUNTER]) || isEmpty(item[FIELD_COUNTER])
                        ? ""
                        : item[FIELD_COUNTER][FIELD_SITE_PATH],
                [FIELD_OFFER_ID_MD5]: item[FIELD_OFFER_ID_MD5],
                [CUSTOM_FIELD_SOURCE_UID]: sourceUniq,
                [FIELD_ACTION_BITS]:
                    isUndefined(item[FIELD_ACTION_BITS]) || isEmpty(item[FIELD_ACTION_BITS])
                        ? ""
                        : item[FIELD_ACTION_BITS],
                [FIELD_SELECT_TYPE]:
                    isUndefined(item[FIELD_SELECT_TYPE]) || isEmpty(item[FIELD_SELECT_TYPE])
                        ? ""
                        : item[FIELD_SELECT_TYPE],
                [FIELD_BID_MULTIPLIER]:
                    isUndefined(item[FIELD_SELECT_TYPE]) || isEmpty(item[FIELD_BID_MULTIPLIER])
                        ? ""
                        : item[FIELD_BID_MULTIPLIER],
                [FIELD_COUNTER_ID]:
                    isUndefined(item[FIELD_COUNTER_ID]) || isEmpty(item[FIELD_COUNTER_ID])
                        ? ""
                        : item[FIELD_COUNTER_ID],
            });
        });

        const profile = sorter[this.state.sortBy](this.state.sortBy, compactProfile);

        return (
            <Table>
                <thead>
                    <tr>
                        {[
                            FIELD_OFFER_NAME,
                            FIELD_SITE_PATH,
                            FIELD_UPDATE_TIME,
                            FIELD_OFFER_ID_MD5,
                            CUSTOM_FIELD_SOURCE_UID,
                            FIELD_ACTION_BITS,
                            FIELD_SELECT_TYPE,
                            FIELD_BID_MULTIPLIER,
                            FIELD_COUNTER_ID,
                        ].map((columnName, key) => {
                            return (
                                <th key={key}>
                                    <div
                                        className="ecom-column-title"
                                        onClick={() => this.handleColumnNameClick(columnName)}
                                    >
                                        {columnName}
                                    </div>
                                </th>
                            );
                        })}
                    </tr>
                </thead>
                <tbody>
                    {profile.map((item, key) => {
                        let offerName = item[FIELD_OFFER_NAME];
                        let offerTargetUrl = isEmpty(item[FIELD_OFFER_TARGET_URL]) ? "" : item[FIELD_OFFER_TARGET_URL];
                        let sitePath = item[FIELD_SITE_PATH];
                        let updateTime = <OptionalTime value={parseInt(item[FIELD_UPDATE_TIME], 10)} />;
                        return (
                            <tr key={key}>
                                <td>
                                    <Link href={offerTargetUrl} target="_blank">
                                        {offerName}
                                    </Link>
                                </td>
                                <td>
                                    <Link href={`http://${sitePath}`} target="_blank">
                                        {sitePath}
                                    </Link>
                                </td>
                                <td>{updateTime}</td>
                                <td>{item[FIELD_OFFER_ID_MD5]}</td>
                                <td>
                                    <div className="ecom-source-uid">{item[CUSTOM_FIELD_SOURCE_UID]}</div>
                                </td>
                                <td>{item[FIELD_ACTION_BITS]}</td>
                                <td>{item[FIELD_SELECT_TYPE]}</td>
                                <td>{item[FIELD_BID_MULTIPLIER]}</td>
                                <td>{item[FIELD_COUNTER_ID]}</td>
                            </tr>
                        );
                    })}
                </tbody>
            </Table>
        );
    }
}

class EcomProfile extends Component {
    splitBySelectType() {
        const profile = this.props.profile[PROFILE_NAME];
        let profileBySt = {};
        profile.forEach((item) => {
            let st = item["select_type"];
            if (!Object.keys(profileBySt).includes(st)) {
                profileBySt[st] = [item];
            } else {
                profileBySt[st].push(item);
            }
        });
        return profileBySt;
    }

    renderProfile() {
        const profile = this.splitBySelectType();
        return (
            <div>
                {Object.keys(profile).map((st, key) => {
                    let stProfile = profile[st];
                    let description = stProfile[0]["select_type_description"];

                    return (
                        <div className="ecom-st-item" key={key}>
                            <h4 className="ecom-st-title">
                                Select Type {st}, {description}
                            </h4>
                            <EcomTable profile={stProfile} sourceUniqs={this.props.sourceUniqs} />
                        </div>
                    );
                })}
            </div>
        );
    }

    render() {
        if (isUndefined(this.props.profile) || isUndefined(this.props.profile[PROFILE_NAME])) {
            return NO_PROFILE;
        } else {
            return (
                <Panel>
                    <div className="ecom-profile-container">{this.renderProfile()}</div>
                </Panel>
            );
        }
    }
}

export default EcomProfile;
