import React from "react";
import { connect } from "react-redux";
import { isUndefined } from "lodash";

import Panel from "@lib/components/panel";
import Json from "@lib/components/json";
import Loading from "@lib/components/loading";

import SearchProfile from "./search-profile";
import Households from "./Households";
import BannersProfile from "./BannersProfile";
import DevicesProfile from "./DevicesProfile";
import TopSitesProfile from "./top-sites";
import UserBans from "./UserBans";
import EcomProfile from "./EcomProfile";
import VisitGoalsProfile from "./visit-goals";
import CategoriesProfile from "./CategoriesProfile";
import ClicksProfile from "./ClicksProfile";
import LoadingSpinner from "../common/LoadingSpinner";

import * as profileNames from "./ProfileNameConstants";
import { getChapter } from "../selectors/profile";
import { getUid } from "../selectors/uid";
import EntityCounters from "./entity-counters/EntityCounters";

const spinner = (
    <div style={{ textAlign: "center", marginTop: "100px" }}>
        <LoadingSpinner size="l" />
    </div>
);

const composeProfile = (profile) => {
    const keywords = profileNames.KEYWORDS_DICT;

    let sections = [];
    if (!isUndefined(profile)) {
        Object.keys(profile).forEach((name) => {
            sections.push({ header: name, content: profile[name] });
        });
    }

    let userProfile = {};
    const keys = Object.keys(keywords);

    for (let i = 0; i < sections.length; i++) {
        let found = false;
        for (let j = 0; j < keys.length; j++) {
            if (keywords[keys[j]].includes(sections[i].header)) {
                if (!Object.keys(userProfile).includes(keys[j])) {
                    userProfile[keys[j]] = {};
                }
                userProfile[keys[j]][sections[i].header] = sections[i].content;
                found = true;
            }
        }

        if (found) {
            continue;
        } else {
            if (!Object.keys(userProfile).includes(profileNames.OTHER)) {
                userProfile[profileNames.OTHER] = {};
            }
            userProfile[profileNames.OTHER][sections[i].header] =
                sections[i].content;
        }
    }

    return userProfile;
};

const UserProfile = connect((state) => ({
    activeProfile: state.bbProfile.activeProfile,
    chapter: getChapter(state),
    activeUid: getUid(state),
}))((props) => {
    const { highlighted, selectTypes, activeProfile, chapter, activeUid } =
        props;

    const renderSearchProfile = (profile, uid, sourceUniqs) => {
        return (
            <SearchProfile
                profile={profile}
                uid={uid}
                highlighted={highlighted}
                selectTypes={selectTypes}
                sourceUniqs={sourceUniqs}
            />
        );
    };

    const renderUserBans = (profile, uid) => {
        let parsed = profile;

        // TODO: Move to api
        if (!isUndefined(profile) && !isUndefined(profile["user-bans"])) {
            parsed = profile["user-bans"].map((item) => {
                const values = item.value.split(":");
                return {
                    time: item.time,
                    domainId: values[0],
                    categoryId: values[1],
                    bannerId: values[2],
                };
            });
        }

        return <UserBans profile={parsed} uid={uid} />;
    };

    const gatherProfileContent = (profile) => {
        let composedProfile;
        let userProfile = {};
        let uid = activeUid;
        let sourceUniqs = [];

        if (!isUndefined(uid)) {
            composedProfile = composeProfile(profile.value);
            if (profile.value && profile.value["source-uniqs"] !== undefined) {
                sourceUniqs = profile.value["source-uniqs"][0].uniqs;
            }
        }

        let knownRenders = {
            [profileNames.SEARCH]: (profile) => {
                return renderSearchProfile(profile, uid, sourceUniqs);
            },
            [profileNames.CRYPTA]: (profile) => {
                return <Households value={profile} />;
            },
            [profileNames.BANNERS]: (profile) => {
                return <BannersProfile profile={profile} uid={uid} />;
            },
            [profileNames.DEVICES]: (profile) => {
                return <DevicesProfile profile={profile} />;
            },
            [profileNames.TOP_SITES]: (profile) => {
                return <TopSitesProfile displayUid={uid} profile={profile} />;
            },
            [profileNames.ABUSES]: (profile) => {
                return renderUserBans(profile, uid);
            },
            [profileNames.ECOM]: (profile) => {
                return (
                    <EcomProfile profile={profile} sourceUniqs={sourceUniqs} />
                );
            },
            [profileNames.GOALS]: (profile) => {
                return <VisitGoalsProfile profile={profile} />;
            },
            [profileNames.CATEGORIES]: (profile) => {
                return <CategoriesProfile profile={profile} uid={uid} />;
            },
            [profileNames.CLICKS]: (profile) => {
                return <ClicksProfile profile={profile} uid={uid} />;
            },
            [profileNames.ENTITY_EXTRACTOR]: (profile) => {
                return <EntityCounters uid={uid} />;
            },
        };

        profileNames.PROFILE_NAMES.forEach((title) => {
            userProfile[title] = {};
        });

        if (isUndefined(composedProfile) || activeProfile.loading) {
            profileNames.PROFILE_NAMES.forEach((title) => {
                userProfile[title] = spinner;
            });
        } else {
            profileNames.PROFILE_NAMES.forEach((title) => {
                if (Object.keys(knownRenders).includes(title)) {
                    let profileToRender = knownRenders[title](
                        composedProfile[title]
                    );
                    userProfile[title] = <div>{profileToRender}</div>;
                } else {
                    userProfile[title] = (
                        <Panel className="Panel-json">
                            <Json data={composedProfile[title]} />
                        </Panel>
                    );
                }
            });
        }

        return userProfile;
    };

    const userProfile = gatherProfileContent(activeProfile);

    return (
        <div className="UserProfile-chapter-content">
            <Loading loaded={!activeProfile.isFetching} size="s">
                {!isUndefined(userProfile) &&
                    !isUndefined(userProfile[chapter]) &&
                    userProfile[chapter]}
            </Loading>
        </div>
    );
});

export default UserProfile;
