import { createAction } from "redux-actions";
import { isUndefined } from "lodash";

import { getEnhancedProfile, getIdentifiers } from "src/utils/apiClient";
import { addErrorNotification } from "../utils/notifications/notificationsState";

const requestActiveProfile = createAction("user profile requested");
const receiveActiveProfile = createAction("user profile received");
const failActiveProfile = createAction("user profile failed");
const requestAdjacentIdentifiers = createAction("adjacent identifiers requested");
const receiveAdjacentIdentifiers = createAction("adjacent identifiers received");
const failAdjacentIdentifiers = createAction("adjacent identifiers failed");
const storeCounterInfo = createAction("counter info stored");

export function fetchUserIdentifiers(idType, idValue) {
    return function (dispatch) {
        dispatch(requestAdjacentIdentifiers({ idType: idType, idValue: idValue }));

        return getIdentifiers(idType, idValue)
            .then((response) => {
                let typedIdentifiers = [{ uid: idValue, type: idType }];

                if (!isUndefined(response)) {
                    typedIdentifiers = [];
                    Object.keys(response).forEach((key) => {
                        response[key].forEach((value) => {
                            typedIdentifiers.push({ uid: value, type: key });
                        });
                    });
                }

                dispatch(
                    receiveAdjacentIdentifiers({
                        idValue: idValue,
                        identifiers: response,
                        typedIdentifiers: typedIdentifiers,
                    })
                );

                return typedIdentifiers;
            })
            .catch((error) => {
                dispatch(
                    failAdjacentIdentifiers({
                        idValue: idValue,
                        identifiers: { [idType]: [idValue] },
                        typedIdentifiers: [{ uid: idValue, type: idType }],
                    })
                );
                if (error.status && error.status !== 404) {
                    dispatch(addErrorNotification(error, "Failed to fetch identifiers"));
                }
            });
    };
}

export function fetchUserProfile(uidType = "BIGB_UID", uid, matching) {
    return function (dispatch) {
        dispatch(requestActiveProfile(matching));

        // TODO: check arguments in API and return BAD_REQUEST
        return getEnhancedProfile(uidType, uid, matching)
            .then((response) => {
                let profile = {};
                if (response) {
                    profile = response.obj;
                }

                dispatch(receiveActiveProfile({ uid: uid, profile: profile }));

                if (!isUndefined(profile["visit-goal"])) {
                    dispatch(storeCounterInfo(profile["visit-goal"]));
                }
            })
            .catch((error) => {
                dispatch(failActiveProfile());
                if (error.status && error.status !== 404) {
                    dispatch(addErrorNotification(error, "Failed to fetch profile"));
                }
            });
    };
}

export {
    requestActiveProfile,
    receiveActiveProfile,
    failActiveProfile,
    requestAdjacentIdentifiers,
    receiveAdjacentIdentifiers,
    failAdjacentIdentifiers,
    storeCounterInfo,
};
