import React, { useEffect, useState } from "react";
import { isEmpty, isUndefined } from "lodash";
import { MD5 } from "object-hash";
import FontAwesome from "react-fontawesome";
import { Button, Checkbox } from "@lib/components/lego";
import Link from "@lib/components/link";
import UidInput from "src/common/UidInput";
import UserProfile from "./UserProfile";
import ControlPanel from "src/common/control-panel";
import Chapters from "src/common/chapters";
import { formatDate } from "@lib/components/time";
import {
    ABUSES,
    BANNERS,
    CATEGORIES,
    CLICKS,
    CRYPTA,
    DEVICES,
    DMP,
    ECOM,
    ENTITY_EXTRACTOR,
    GOALS,
    MOBILE,
    OTHER,
    SEARCH,
    TOP_SITES,
} from "./ProfileNameConstants";

import { connect } from "react-redux";

import "./style.scss";

import { initProfileSettings } from "src/bb-profile/settings/settingsActions";
import UidTypeSelectInput from "src/bb-profile/controls/UidTypeSelectInput";
import {
    ALL_TO_BB_PROFILE,
    BB_VIEWER_URL,
    COMMON_DUID,
    COMMON_IDS_FOR_PROFILES,
    COMMON_TO_BIGB_API_ID_MAP,
    COMMON_TO_BIGB_ID_MAP,
    COMMON_YANDEXUID,
} from "src/common/Names";
import { fetchSelectTypes } from "src/ad/selectTypes/actions";
import { invalidateCommonProfiles } from "src/bb-profile/common-profiles/actions";
import { fetchUserIdentifiers, fetchUserProfile } from "src/bb-profile/actions";
import { getUid, getUidType } from "src/selectors";
import { getChapter, getMatching } from "src/selectors";
import { setInQuery, setUid } from "src/actions/router";
import { dropAllNotifications } from "../utils/notifications/notificationsState";
import AdjacentUidsNavigation from "./adjacent-uids";
import { getActiveAdjacentUid, getActiveAdjacentUidType } from "./selectors";
import { useTranslation } from "react-i18next";
import { reachGoal } from "../metrika";
import { fetchEntityCounters } from "./entity-counters/actions";

const getBbProfileElementHeight = () => {
    let adContainerElement = document
        .getElementsByClassName("BbProfile-content")
        .item(0);
    let elementHeight = window
        .getComputedStyle(adContainerElement, null)
        .getPropertyValue("height");
    return parseFloat(elementHeight.substring(0, elementHeight.length - 2));
};

const BbProfile = connect((state) => ({
    uid: getUid(state),
    uidType: ALL_TO_BB_PROFILE[getUidType(state)],
    matching: getMatching(state),
    chapter: getChapter(state),
    selectTypes: state.selectTypes.items,
    identifiers: state.bbProfile.identifiers,
    activeProfile: state.bbProfile.activeProfile,
    sources: state.sources,
    activeAdjacentUid: getActiveAdjacentUid(state),
    activeAdjacentUidType: getActiveAdjacentUidType(state),
}))((props) => {
    const {
        uid,
        uidType,
        matching,
        chapter,
        selectTypes,
        activeProfile,
        activeAdjacentUid,
        activeAdjacentUidType,
        dispatch,
    } = props;

    const [queryIdHighlight, setQueryIdHighlight] = useState();
    const [viewerContentHeight, setViewerContentHeight] = useState(0);

    const { t } = useTranslation("bbprofile");

    useEffect(() => {
        setViewerContentHeight(getBbProfileElementHeight());

        dispatch(dropAllNotifications());
        dispatch(invalidateCommonProfiles());
    }, [uid]);

    useEffect(() => {
        if (location && location.hash) {
            setQueryIdHighlight(location.hash);
        }
    }, []);

    useEffect(() => {
        dispatch(initProfileSettings());
        dispatch(
            setInQuery("uidType", isUndefined(uidType) ? "yandexuid" : uidType)
        );
        dispatch(fetchUserIdentifiers(uidType, uid));
        dispatch(fetchSelectTypes());

        setViewerContentHeight(getBbProfileElementHeight());
    }, []);

    useEffect(() => {
        if (isEmpty(activeAdjacentUid) || isEmpty(activeAdjacentUidType)) {
            dispatch(
                fetchUserProfile(
                    COMMON_TO_BIGB_API_ID_MAP[uidType],
                    uid,
                    matching
                )
            );
            dispatch(
                fetchEntityCounters(uid, COMMON_TO_BIGB_API_ID_MAP[uidType])
            );
        }
    }, []);

    useEffect(() => {
        if (!isEmpty(activeAdjacentUid) && !isEmpty(activeAdjacentUidType)) {
            dispatch(
                fetchUserProfile(
                    COMMON_TO_BIGB_API_ID_MAP[activeAdjacentUidType],
                    activeAdjacentUid,
                    matching
                )
            );
            dispatch(
                fetchEntityCounters(
                    activeAdjacentUid,
                    COMMON_TO_BIGB_API_ID_MAP[activeAdjacentUidType]
                )
            );
        }
    }, [activeAdjacentUid]);

    const onUidInputChange = (uidValue) => {
        dispatch(setUid(uidValue));
    };

    const handleGetProfile = () => {
        dispatch(invalidateSources());
        dispatch(dropAllNotifications());

        dispatch(fetchUserIdentifiers(uidType, uid));
    };

    const onRefresh = () => {
        handleGetProfile();
        reachGoal("refreshBbProfile");
    };

    const onMatchingChanged = () => {
        dispatch(
            setInQuery("matching", (!!!+matching + false).toString(), false)
        );
        dispatch(reachGoal("changeMatchingOption"));
    };

    const renderBbViewerLink = () => {
        const queryParams = [
            [COMMON_TO_BIGB_ID_MAP[uidType], uid],
            ["glue", matching],
        ];

        if (COMMON_TO_BIGB_ID_MAP[uidType] === COMMON_DUID) {
            queryParams.push([COMMON_TO_BIGB_ID_MAP[COMMON_YANDEXUID], uid]);
        }

        const url =
            BB_VIEWER_URL +
            "?" +
            queryParams.map((queryParams) => queryParams.join("=")).join("&");

        return (
            <div className="BbProfile-bb-viewer-link">
                <span className="BbProfile-bb-viewer-link-url">
                    <Link
                        href={url}
                        target="_blank"
                        onClick={() => dispatch(reachGoal("clickBbViewer"))}
                    >
                        bb-viewer
                    </Link>
                </span>
                <FontAwesome name="external-link" />
            </div>
        );
    };

    const onChapterSelect = (chapterValue) => {
        dispatch(setInQuery("chapter", chapterValue));
        dispatch(reachGoal("chapterSelected_" + chapterValue));
    };

    const onSelectUidType = (value) => {
        dispatch(setInQuery("uidType", value, false));
        dispatch(reachGoal("profileUidTypeSelected"));
    };

    const renderUserProfile = () => {
        const loaded = activeProfile.didFetch || false;

        return (
            <UserProfile
                uid={uid}
                loaded={loaded}
                highlighted={queryIdHighlight}
                selectTypes={selectTypes}
            />
        );
    };

    const uidCreationTimeLabel =
        isNaN(uid) || isEmpty(uid)
            ? "Unavailable"
            : `UID created ${formatDate(parseInt(uid.slice(-10), 10))}`;

    const chapters = [
        { value: SEARCH, title: t("chapters.search") },
        { value: CLICKS, title: t("chapters.clicks") },
        { value: ECOM, title: t("chapters.ecom") },
        { value: CATEGORIES, title: t("chapters.categories") },
        { value: GOALS, title: t("chapters.goals") },
        { value: TOP_SITES, title: t("chapters.topSites") },
        { value: CRYPTA, title: t("chapters.households") },
        { value: ABUSES, title: t("chapters.abuses") },
        { value: BANNERS, title: t("chapters.banners") },
        { value: DEVICES, title: t("chapters.devices") },
        { value: DMP, title: t("chapters.dmp") },
        { value: MOBILE, title: t("chapters.mobile") },
        { value: ENTITY_EXTRACTOR, title: t("chapters.entity_extractor") },
        { value: OTHER, title: t("chapters.other") },
    ];

    const controls = (
        <div>
            <table>
                <tbody>
                    <tr>
                        <td colSpan={4}>
                            <div
                                className="uid-creation-time"
                                style={{ marginBottom: "10px" }}
                            >
                                {uidCreationTimeLabel}
                            </div>
                        </td>
                    </tr>
                    <tr>
                        <td className="td-label">
                            <UidTypeSelectInput
                                key={MD5(COMMON_IDS_FOR_PROFILES)}
                                placeholder="[bigb id type]"
                                items={COMMON_IDS_FOR_PROFILES}
                                value={uidType}
                                onChange={onSelectUidType}
                            />
                        </td>
                        <td colSpan={3}>
                            <UidInput value={uid} onChange={onUidInputChange} />
                        </td>
                    </tr>
                    <tr>
                        <td>
                            <Checkbox
                                label={t("withMatching")}
                                size="m"
                                view="default"
                                onChange={onMatchingChanged}
                                checked={!!+matching}
                            />
                        </td>
                        <td>
                            <Button view="action" onClick={onRefresh} size="m">
                                {t("refresh")}
                            </Button>
                        </td>
                    </tr>
                </tbody>
            </table>

            <Chapters
                values={chapters}
                onSelect={onChapterSelect}
                active={chapter}
            />

            {renderBbViewerLink()}
        </div>
    );

    return (
        <div className="BbProfile">
            <ControlPanel
                width="353px"
                outerHeight={viewerContentHeight}
                content={controls}
                onRefresh={handleGetProfile}
            />

            <div className="BbProfile-content">
                <AdjacentUidsNavigation />
                {renderUserProfile()}
            </div>
        </div>
    );
});

export default BbProfile;
