import React, { useState, useEffect } from "react";
import { isUndefined, isEmpty, values as _values, chain as _chain } from "lodash";
import { connect } from "react-redux";
import { Button, Modal } from "@lib/components/lego";
import Panel from "@lib/components/panel";
import Table from "@lib/components/table";
import Link from "@lib/components/link";
import OptionalTime from "@lib/components/time";

import ModalHeader from "../../common/modal-header";
import CommonProfileModal from "./CommonProfileModal";
import { prettyPreJson } from "../../utils/prettifiers";
import { fetchCommonProfile } from "src/bb-profile/common-profiles/actions";

import "./style.scss";
import { reachGoal } from "../../metrika";

const SearchProfile = connect((state) => ({
    commons: state.commonProfiles,
}))((props) => {
    const { uid, profile, commons, dispatch, highlighted, selectTypes, sourceUniqs } = props;

    const [modalVisible, setModalVisibility] = useState(false);
    const [modalContent, setModalContent] = useState(null);
    const [modalHeader, setModalHeader] = useState("");
    const [modalInfo, setModalInfo] = useState("");

    const [commonProfileModalVisible, setCommonProfileModalVisibility] = useState(false);
    const [commonProfileContent, setCommonProfileContent] = useState(null);

    const getCommonProfiles = (userProfile) => {
        if (userProfile && userProfile["search-text"]) {
            const profiles = userProfile["search-text"];

            const commonProfileIds = new Set(
                profiles
                    .filter((profile) => {
                        return !isEmpty(profile["common_id"]);
                    })
                    .map((profile) => profile["common_id"])
            );

            commonProfileIds.forEach((id) => {
                dispatch(fetchCommonProfile(id));
            });
        }
    };

    useEffect(() => {
        getCommonProfiles(profile);
    }, []);

    const onOpenModal = (content, header, info) => {
        setModalVisibility(true);
        setModalHeader(header);
        setModalInfo(info);
        setModalContent(content);
        dispatch(reachGoal("checkSelectTypeJSON"));
    };

    const onCloseModal = () => {
        setModalVisibility(false);
        setModalContent(null);
        setModalHeader("");
        setModalInfo(null);
    };

    const onCommonProfileModalOpen = (content) => {
        setCommonProfileModalVisibility(true);
        setCommonProfileContent(content);
    };

    const onCommonProfileModalClose = () => {
        setCommonProfileModalVisibility(false);
        setCommonProfileContent(null);
    };

    const renderSelectTypeTable = (profiles, itemNames) => {
        const weight = profiles.weight;
        const descriptionProfile = profiles[Object.keys(profiles)[0]];
        const selectType = descriptionProfile["select_type"];

        const prettySelectType = <div className="SearchProfile-select-type-json">{prettyPreJson(profiles)}</div>;

        const queryIdField = "query_id";
        return (
            <Panel key={selectType}>
                <h4 className="select-type-title">
                    Select type {selectType}, {descriptionProfile["select_type_description"]} (weight {weight}){" "}
                    <Button
                        size="m"
                        view="action"
                        title="JSON"
                        onClick={() =>
                            onOpenModal(
                                prettySelectType,
                                descriptionProfile["select_type_description"],
                                `Select type ${selectType}`
                            )
                        }
                    >
                        JSON
                    </Button>
                </h4>
                <Table>
                    <thead>
                        <tr style={{ textAlign: "center" }}>
                            {itemNames.map((itemName) => {
                                return <td key={itemName}>{itemName}</td>;
                            })}
                        </tr>
                    </thead>
                    <tbody>
                        {profiles.map((profile, index) => {
                            let hlClassname = "";

                            if (!isUndefined(highlighted)) {
                                hlClassname =
                                    highlighted.replace(/#query_id/, "") === profile[queryIdField] ? "hl-tr" : "";
                            }
                            return (
                                <tr key={index} className={hlClassname}>
                                    {itemNames.map(function (itemName, i) {
                                        if (itemName === "categories") {
                                            return (
                                                <td title={itemName} key={itemName}>
                                                    {JSON.stringify(profile[itemName], undefined, 2)}
                                                </td>
                                            );
                                        } else if (itemName === "parent_domain") {
                                            return (
                                                <td title={itemName} key={itemName}>
                                                    <Link href={"http://" + profile[itemName]} target="_blank">
                                                        {profile[itemName]}
                                                    </Link>
                                                </td>
                                            );
                                        } else if (itemName === "create_time" || itemName === "time") {
                                            return (
                                                <td title={itemName} key={itemName}>
                                                    <OptionalTime value={parseInt(profile[itemName], 10)} />
                                                </td>
                                            );
                                        } else if (itemName === queryIdField) {
                                            const itemId = queryIdField + profile[itemName];

                                            return (
                                                <td title={itemName} key={itemName}>
                                                    <span id={itemId} />
                                                    {profile[itemName]}
                                                </td>
                                            );
                                        } else if (itemName === "source uid") {
                                            let index = profile["source_uniq_index"];
                                            let sourceUniq = sourceUniqs[index].user_id;
                                            return (
                                                <td title={itemName} key={itemName}>
                                                    <div
                                                        style={{
                                                            color: "#228b22",
                                                        }}
                                                    >
                                                        {sourceUniq}
                                                    </div>
                                                </td>
                                            );
                                        } else if (
                                            itemName === "common_id" &&
                                            !isUndefined(profile[itemName]) &&
                                            !isEmpty(profile[itemName])
                                        ) {
                                            let commonId = profile[itemName];
                                            return (
                                                <td title={itemName} key={itemName}>
                                                    <div
                                                        className="common-id"
                                                        onClick={() =>
                                                            onCommonProfileModalOpen(commons[commonId].profile)
                                                        }
                                                    >
                                                        {commonId}
                                                    </div>
                                                </td>
                                            );
                                        } else if (itemName === "common_name" && profile["common_id"]) {
                                            let value = "";
                                            let commonProfile = commons[profile["common_id"]];
                                            if (!isUndefined(commonProfile)) {
                                                value = commons[profile["common_id"]]["profile"]["description"];
                                            }
                                            return <td key={itemName}>{value}</td>;
                                        } else {
                                            return (
                                                <td title={itemName} key={itemName}>
                                                    {profile[itemName]}
                                                </td>
                                            );
                                        }
                                    })}
                                </tr>
                            );
                        })}
                    </tbody>
                </Table>
            </Panel>
        );
    };

    const renderSearchTextItemsTables = (profiles, itemNames) => {
        // Split profile by select_type
        let profilesBySelectType = {};
        profiles.forEach((profile) => {
            const selectType = profile["select_type"];

            if (!Object.keys(profilesBySelectType).includes(selectType)) {
                profilesBySelectType[selectType] = [];
            }

            profilesBySelectType[selectType].push(profile);
            profilesBySelectType[selectType].weight = selectTypes[selectType].weight;
        });

        // Displaying these names are redundant
        ["id", "name", "select_type", "select_type_description", "collector_pathes", "collector_path_bits"].forEach(
            (nameToHide) => {
                itemNames.splice(itemNames.indexOf(nameToHide), 1);
            }
        );

        // Set proper order of columns
        let priorColumnNames = [
            "value",
            "parent_domain",
            "query_id",
            "source uid",
            "common_id",
            "common_name",
            "categories",
        ];
        itemNames.forEach((itemName) => {
            if (!priorColumnNames.includes(itemName)) {
                priorColumnNames.push(itemName);
            }
        });

        itemNames = priorColumnNames;
        const selectTypeTables = _chain(_values(profilesBySelectType))
            .sortBy((selectTypeProfile) => selectTypeProfile.weight)
            .reverse()
            .map((selectTypeProfile) => {
                return renderSelectTypeTable(selectTypeProfile, itemNames);
            })
            .value();

        return <div>{selectTypeTables}</div>;
    };

    let itemNames = [];
    let profiles = [];

    if (isUndefined(profile)) {
        return <div className="empty-content">User has no search-text profile</div>;
    } else {
        profile["search-text"].forEach((itemList) => {
            profiles.push(itemList);
            Object.keys(itemList).forEach((item) => {
                if (!itemNames.includes(item)) {
                    itemNames.push(item);
                }
            });
        });
        itemNames.push("source uid");
    }

    return (
        <div>
            {renderSearchTextItemsTables(profiles, itemNames)}

            {!isEmpty(modalContent) && (
                <Modal theme="normal" visible={modalVisible} onClose={onCloseModal}>
                    <ModalHeader title={modalHeader} info={modalInfo} onClose={onCloseModal} hasClose />
                    {modalContent}
                </Modal>
            )}

            {!isEmpty(commonProfileContent) && (
                <CommonProfileModal
                    visible={commonProfileModalVisible}
                    content={modalContent}
                    onClose={onCommonProfileModalClose}
                />
            )}
        </div>
    );
});

export default SearchProfile;
