import React, { Component } from "react";
import { connect } from "react-redux";

import Panel from "@lib/components/panel";
import Table from "@lib/components/table";
import Link from "@lib/components/link";
import OptionalTime from "@lib/components/time";

import ModalClosable from "../../common/ModalClosable";
import ActionButton from "../../common/ActionButton";
import { sortByInt, sortAB } from "../../utils/sorters";
import { VISIT_GOAL, VISIT_GOAL_COUNTER_ID, VISIT_GOAL_COUNTER_INFO } from "../ProfileNameConstants";
import { prettyPreJson } from "../../utils/prettifiers";

import "./style.scss";
import { reachGoal } from "../../metrika";

const NO_PROFILE = <div className="empty-content">User has no visit-goal profile</div>;

const FIELD_GOAL_INFO = "goal_info";
const FIELD_GOAL_TYPE = "goal_type";
const FIELD_COUNTER_ID = VISIT_GOAL_COUNTER_ID;
const FIELD_NAME = "name";
const FIELD_TIME = "time";
const FIELD_SITE = "site";
const FIELD_URLS = "urls";

const TH_GOAL_INFO = "goal_type : goal_id / counter_id";
const TH_GOAL_NAME = "name";
const TH_TIME = "time";
const TH_SITE = "site";
const TH_URLS = "urls";
const TH_COUNTER_INFO = VISIT_GOAL_COUNTER_INFO;

@connect((state) => ({
    counterInfo: state.counterInfo,
}))
export default class VisitGoalsProfile extends Component {
    constructor() {
        super();
        this.handleColumnNameClick = this.handleColumnNameClick.bind(this);
        this.handleShowCounterInfoModal = this.handleShowCounterInfoModal.bind(this);

        this.state = {
            sortBy: TH_TIME,
            descending: {},
            showCounterInfoModal: false,
            counterInfoModalContent: undefined,
            counterInfoModalHeading: "",
        };
    }

    componentDidMount() {
        let descending = {};
        [TH_GOAL_INFO, TH_GOAL_NAME, TH_SITE, TH_URLS].forEach((item) => {
            descending[item] = true;
        });

        [TH_TIME].forEach((item) => {
            descending[item] = true;
        });

        this.setState({ descending: descending });
    }

    handleColumnNameClick(columnName) {
        let descending = Object.assign({}, this.state.descending);
        descending[columnName] =
            this.state.sortBy === columnName ? !this.state.descending[columnName] : this.state.descending[columnName];
        this.setState({
            sortBy: columnName,
            descending: descending,
        });
    }

    handleShowCounterInfoModal(counterInfo, heading) {
        this.setState({
            showCounterInfoModal: true,
            counterInfoModalContent: counterInfo,
            counterInfoModalHeading: heading,
        });
        this.props.dispatch(reachGoal("checkCounterInfo"));
    }

    handleHideCounterInfoModal() {
        this.setState({
            showCounterInfoModal: false,
            counterInfoModalContent: undefined,
            counterInfoModalHeading: "",
        });
    }

    renderProfile() {
        let compactProfile = [];

        const sorter = {
            [TH_GOAL_INFO]: (what) => {
                return sortAB(FIELD_GOAL_INFO, what, this.state.descending[TH_GOAL_INFO]);
            },
            [TH_GOAL_NAME]: (what) => {
                return sortAB(FIELD_NAME, what, this.state.descending[TH_GOAL_NAME]);
            },
            [TH_TIME]: (what) => {
                return sortByInt(FIELD_TIME, what, this.state.descending[TH_TIME]);
            },
            [TH_SITE]: (what) => {
                return sortAB(FIELD_SITE, what, this.state.descending[TH_SITE]);
            },
            [TH_URLS]: (what) => {
                return sortAB(FIELD_URLS, what, this.state.descending[TH_URLS]);
            },
        };

        this.props.profile[VISIT_GOAL].forEach((item) => {
            let info = item[FIELD_GOAL_INFO];
            let goal;
            let name;
            let site;
            let urls;
            let goalId = item["value"];
            let counterId;
            if (info === null || typeof info === "undefined" || info.length === 0) {
                goal = `: ${goalId} /`;
                name = "";
                site = "";
                urls = "";
                counterId = null;
            } else {
                goal = `${info[FIELD_GOAL_TYPE]} : ${goalId} / ${info[FIELD_COUNTER_ID]}`;
                name = info[FIELD_NAME];
                site = info[FIELD_SITE];
                urls = info[FIELD_URLS].join("\n");
                counterId = info[FIELD_COUNTER_ID] ? info[FIELD_COUNTER_ID] : "";
            }

            compactProfile.push({
                [VISIT_GOAL_COUNTER_ID]: counterId,
                [FIELD_GOAL_INFO]: goal,
                [FIELD_NAME]: name,
                [FIELD_URLS]: urls,
                [FIELD_SITE]: site,
                [FIELD_TIME]: item[FIELD_TIME],
            });
        });

        const profile = sorter[this.state.sortBy](compactProfile);

        return (
            <div>
                <ModalClosable
                    isOpen={this.state.showCounterInfoModal}
                    heading={this.state.counterInfoModalHeading}
                    handleHideModal={() => this.handleHideCounterInfoModal()}
                    content={prettyPreJson(this.state.counterInfoModalContent)}
                />

                <Table>
                    <thead>
                        <tr>
                            {[TH_GOAL_INFO, TH_COUNTER_INFO, TH_GOAL_NAME, TH_TIME, TH_SITE, TH_URLS].map(
                                (columnName, key) => {
                                    let value;
                                    if (columnName === TH_COUNTER_INFO) {
                                        value = <div>{columnName}</div>;
                                    } else {
                                        value = (
                                            <div
                                                className="goal-column-title"
                                                onClick={() => this.handleColumnNameClick(columnName)}
                                            >
                                                {columnName}
                                            </div>
                                        );
                                    }
                                    return <th key={key}>{value}</th>;
                                }
                            )}
                        </tr>
                    </thead>
                    <tbody>
                        {profile.map((item, key) => {
                            const counterId = item[VISIT_GOAL_COUNTER_ID];
                            const showCounterInfoButton = counterId ? (
                                <ActionButton
                                    size="s"
                                    title={counterId.toString()}
                                    displayTitle={counterId.toString()}
                                    onClick={() =>
                                        this.handleShowCounterInfoModal(
                                            this.props.counterInfo[counterId]
                                                ? this.props.counterInfo[counterId]
                                                : `No counter info for ${counterId}`,
                                            `Counter ID ${counterId}`
                                        )
                                    }
                                />
                            ) : (
                                ""
                            );

                            return (
                                <tr key={key}>
                                    <td>{item[FIELD_GOAL_INFO]}</td>
                                    <td>{showCounterInfoButton}</td>
                                    <td>
                                        <div className="goal-wide-field">{item[FIELD_NAME]}</div>
                                    </td>
                                    <td>
                                        <OptionalTime value={parseInt(item[FIELD_TIME], 10)} />
                                    </td>
                                    <td>
                                        <div className="goal-wide-field">
                                            <Link href={`http://${item[FIELD_SITE]}`} target="_blank">
                                                {item[FIELD_SITE]}
                                            </Link>
                                        </div>
                                    </td>
                                    <td>
                                        <div className="goal-wide-field">{item[FIELD_URLS]}</div>
                                    </td>
                                </tr>
                            );
                        })}
                    </tbody>
                </Table>
            </div>
        );
    }

    render() {
        const condition = this.props.profile && this.props.profile[VISIT_GOAL];
        return (
            <Panel>
                {condition ? <div className="visit-goals-container">{this.renderProfile()}</div> : NO_PROFILE}
            </Panel>
        );
    }
}
