import React, { useLayoutEffect, useState } from "react";
import FontAwesome from "react-fontawesome";

import "./style.scss";
import { useDispatch } from "react-redux";
import { reachGoal } from "../../metrika";

const HEIGHT_BY_WINDOW = "by_window";

/**
 * Main Portal setting panel. It is to become common management tool for the Portal.
 * For now it is shown only in testing environment
 */
const ControlPanel = (props) => {
    const { width, widthWhenHidden, outerHeight, heightMode, onChange, onRefresh, content } = props;

    const [windowHeight, setWindowHeight] = useState(0);
    const [visible, setVisibility] = useState(true);

    const dispatch = useDispatch();

    useLayoutEffect(() => {
        function updateSize() {
            setWindowHeight(window.innerHeight);
        }
        window.addEventListener("resize", updateSize, true);
        updateSize();

        return () => window.removeEventListener("resize", updateSize);
    }, []);

    const onSetVisibility = () => {
        setVisibility(!visible);

        if (onChange !== null) {
            onChange();
        }

        dispatch(reachGoal("hideOrShowControlPanel"));

        localStorage.setItem("controlPanelVisibility", visible ? "hidden" : "visible");
    };

    let actualHeight;
    let actualWidth = visible ? width : widthWhenHidden;
    let contentDisplay = visible ? {} : { display: "none" };

    switch (heightMode) {
        case HEIGHT_BY_WINDOW:
            actualHeight = outerHeight;
            break;
        default:
            actualHeight = Math.max(windowHeight, outerHeight) + 20;
            break;
    }

    return (
        <div className="ControlPanel" style={{ height: actualHeight + "px", width: actualWidth }}>
            <div>
                <ExpandPanelButton toCollapse={visible} onClick={onSetVisibility} />
                <RefreshProfileButton displayStyle={!visible ? {} : { display: "none" }} onClick={onRefresh} />
                <div style={contentDisplay}>{content}</div>
            </div>
        </div>
    );
};

ControlPanel.defaultProps = {
    width: "470px",
    widthWhenHidden: "50px",
    onChange: null,
};

const ExpandPanelButton = (props) => {
    const { toCollapse, onClick } = props;

    return (
        <div className="ControlPanel-glyph" onClick={onClick}>
            <FontAwesome name={toCollapse ? "chevron-left" : "chevron-right"} />
        </div>
    );
};

const RefreshProfileButton = (props) => {
    const { displayStyle, onClick } = props;

    return (
        <div className="ControlPanel-refresh-profile" style={displayStyle} onClick={onClick} title="Refresh profile">
            <FontAwesome name="repeat" />
        </div>
    );
};

export default ControlPanel;
