import React, { Component } from "react";
import { connect } from "react-redux";
import FontAwesome from "react-fontawesome";
import { debounce, isEmpty } from "lodash";
import { Map, Marker, MarkerLayout } from "yandex-map-react";

import { TextInput } from "@lib/components/lego";
import Loading from "@lib/components/loading";
import { getCenter, getRadius, getZoom, pairToPoint, pointToPair } from "@lib/components/map";

import createSwagger, { getHomeLocation, getWorkLocation } from "src/utils/apiClient";
import ActionButton from "src/common/ActionButton";
import ControlPanel from "src/common/control-panel";
import Chapters from "src/common/chapters";
import GeoInfoBar from "./GeoInfoBar";
import UidInput from "src/common/UidInput";

import { getUid, getUidType } from "src/selectors/uid";
import { fetchUserIdentifiers } from "src/bb-profile/actions";
import { fetchGeo, fetchGeoIfNeeded, invalidateGeo } from "src/geo/actions";
import { dropQueryParam, setUid } from "src/actions/router";
import { getGeoLocations } from "src/selectors/geo";
import { getProfileIdentifiers } from "src/selectors/profile";

import "./style.scss";
import { getProfileIdentifiersLoading } from "../selectors/profile";

const BAR_HEIGHT = 50;
const CONTROL_BAR_WIDTH = 300;
const CONTROL_BAR_HIDDEN_WIDTH = 50;

const mapFieldCryptaGeolocation = "krypta-geolocation";
const mapFieldCryptaRegularLocation = "krypta-regular-location";

function convertGeoToNormal(points) {
    return points.map((point) => {
        return point
            .map((coordinate) => {
                return coordinate / 10000000 - 180;
            })
            .reverse();
    });
}

function getExtraWidth() {
    return localStorage.getItem("controlPanelVisibility") === "visible" ? CONTROL_BAR_WIDTH : CONTROL_BAR_HIDDEN_WIDTH;
}

const geoMarkerRegular = (
    <div className="geo-marker geo-marker-regular">
        <div className="geo-marker-inner" />
    </div>
);
const geoMarkerActual = (
    <div className="geo-marker geo-marker-actual">
        <div className="geo-marker-inner" />
    </div>
);
const geoMarkerHome = (
    <div className="geo-homework-marker">
        <FontAwesome name="home" />
    </div>
);
const geoMarkerWork = (
    <div className="geo-homework-marker">
        <FontAwesome name="briefcase" />
    </div>
);

// TODO: make Geo chapters not blink

@connect(
    (state) => ({
        uid: getUid(state),
        uidType: getUidType(state),
        locations: getGeoLocations(state),
        identifiers: getProfileIdentifiers(state),
        identifiersLoading: getProfileIdentifiersLoading(state),
    }),
    (dispatch) => ({
        setUid: (uid) => dispatch(setUid(uid)),
        fetchGeoIfNeeded: debounce((uidType, uid) => dispatch(fetchGeoIfNeeded(uidType, uid), 800)),
        dispatch: dispatch,
    })
)
export default class Geo extends Component {
    constructor() {
        super();
        this.handleRegularPointClick = this.handleRegularPointClick.bind(this);
        this.handleResize = this.handleResize.bind(this);
        this.handleUidInput = this.handleUidInput.bind(this);
        this.handleGetGeoProfile = this.handleGetGeoProfile.bind(this);
        this.handleControlPanelChange = this.handleControlPanelChange.bind(this);

        this.apiClient = createSwagger();

        this.state = {
            pointInfo: undefined,
            extraWidth: getExtraWidth(),
        };
    }

    componentDidMount() {
        if (isEmpty(this.props.identifiers)) {
            this.props.dispatch(fetchUserIdentifiers(this.props.uidType, this.props.uid));
        }

        if (this.props.uid) {
            this.props.fetchGeoIfNeeded(this.props.uidType, this.props.uid);
        }
        window.addEventListener("resize", this.handleResize, true);
        this.props.dispatch(dropQueryParam("chapter"));
    }

    componentWillUnmount() {
        window.removeEventListener("resize", this.handleResize, true);
    }

    componentDidUpdate(prevProps) {
        if (this.props.uid !== prevProps.uid && !isEmpty(prevProps.uid)) {
            this.props.fetchGeoIfNeeded(this.props.uidType, this.props.uid);
        }
    }

    handleResize() {
        this.forceUpdate();
    }

    handleGetGeoProfile(uid) {
        this.getGeoProfile(uid);
    }

    getGeoProfile(uid) {
        getHomeLocation(uid).then((result) => {
            this.setState({ home: result });
        });
        getWorkLocation(uid).then((result) => {
            this.setState({ work: result });
        });

        this.props.dispatch(invalidateGeo());
        this.props.dispatch(fetchGeo(this.props.uidType, uid));
    }

    handleRegularPointClick(updateTime, lat, lon, type) {
        this.setState({
            pointInfo: {
                updateTime: updateTime,
                latitude: lat,
                longitude: lon,
                type: type,
            },
        });
    }

    handleUidInput(uid) {
        this.props.dispatch(setUid(uid));
    }

    handleControlPanelChange() {
        this.setState({ extraWidth: getExtraWidth() });
    }

    render() {
        const locations = this.props.locations;
        let regular = [];
        let actual = [];
        let actualUpdateTime = "";

        if (
            typeof locations[mapFieldCryptaRegularLocation] !== "undefined" &&
            Object.keys(locations[mapFieldCryptaRegularLocation]).length > 0
        ) {
            regular = locations[mapFieldCryptaRegularLocation]["data"];
        }

        if (
            typeof locations[mapFieldCryptaGeolocation] !== "undefined" &&
            Object.keys(locations[mapFieldCryptaGeolocation]).length > 0
        ) {
            const rawActual = locations[mapFieldCryptaGeolocation].value.split(",").map((point) => {
                return point.split(":");
            });
            actual = convertGeoToNormal(rawActual);
            actualUpdateTime = locations[mapFieldCryptaGeolocation].time;
        }

        const allPoints = regular
            .map((point) => {
                return [point.latitude, point.longitude];
            })
            .concat(actual);

        const chapters = isEmpty(this.props.identifiers.yandexuid)
            ? []
            : this.props.identifiers.yandexuid.map((item) => ({ value: item, title: item }));

        let controls = (
            <div className="Geo-controls">
                <table className="geo-controls-table">
                    <tbody>
                        <tr>
                            <td className="td-label">ID</td>
                            <td>
                                <TextInput
                                    view="default"
                                    size="m"
                                    placeholder="uid"
                                    value={this.props.uid}
                                    onChange={(event) => this.handleUidInput(event.target.value)}
                                    onClearClick={() => this.handleUidInput("")}
                                    hasClear
                                />
                            </td>
                        </tr>
                        <tr>
                            <td colSpan={2}>
                                <div style={{ float: "right", marginTop: "6px" }}>
                                    <ActionButton
                                        onClick={() => this.handleGetGeoProfile(this.props.uid)}
                                        title="Get Geo Profile"
                                        displayTitle="Get Geo"
                                    />
                                </div>
                            </td>
                        </tr>
                    </tbody>
                </table>

                <Chapters values={chapters} onSelect={this.props.setUid} active={this.props.uid} />
            </div>
        );

        const mapWidth = window.innerWidth - this.state.extraWidth;

        return (
            <div className="geo-viewer-content" style={{ overflow: "hidden" }}>
                <table style={{ width: "100%" }}>
                    <tbody>
                        <tr>
                            <td style={{ margin: "0", width: "1%" }}>
                                <ControlPanel
                                    width="300px"
                                    heightMode="by_window"
                                    outerHeight={window.innerHeight - BAR_HEIGHT}
                                    content={controls}
                                    onRefresh={() => this.handleGetGeoProfile(this.props.uid)}
                                    onChange={this.handleControlPanelChange}
                                />
                            </td>
                            <td>
                                <div className="geo-container">
                                    <Loading loaded={!this.props.identifiersLoading} size="m">
                                        {isEmpty(locations) ? (
                                            <div
                                                className="empty-content"
                                                style={{
                                                    height: window.innerHeight - BAR_HEIGHT * 2,
                                                }}
                                            >
                                                User has no actual or regular coordinates
                                            </div>
                                        ) : (
                                            <UserMap
                                                points={locations}
                                                pointInfo={this.state.pointInfo}
                                                width={mapWidth}
                                                height={window.innerHeight - BAR_HEIGHT * 2}
                                                onRegularClick={this.handleRegularPointClick}
                                                uid={this.props.uid}
                                                regular={regular}
                                                actual={actual}
                                                home={this.state.home}
                                                work={this.state.work}
                                                actualUpdateTime={actualUpdateTime}
                                                allPoints={allPoints}
                                            />
                                        )}
                                    </Loading>
                                </div>
                            </td>
                        </tr>
                    </tbody>
                </table>
            </div>
        );
    }
}

const mapState = {
    controls: ["default"],
};

class UserMap extends Component {
    componentDidUpdate(nextProps) {
        if (nextProps !== this.props) {
            /**
             * I couldn't find out how to manipulate Yandex.Maps React component,
             * so I just change DOM directly
             */
            let mapsWrapperElement = document.getElementsByClassName("ymaps-2-1-60-map").item(0);

            if (mapsWrapperElement === null) {
                return;
            }

            let mapsElement = document
                .getElementsByClassName(
                    "ymaps-2-1-60-map ymaps-2-1-60-i-ua_js_yes ymaps-2-1-60-map-bg-ru ymaps-2-1-60-islets_map-lang-ru"
                )
                .item(0);

            if (mapsElement === null) {
                return;
            }

            if (this.props.width !== nextProps.width) {
                if (mapsWrapperElement !== null && mapsElement !== null) {
                    mapsWrapperElement.style.width = nextProps.width + "px";
                    mapsElement.style.width = nextProps.width + "px";
                }
            }

            if (this.props.height !== nextProps.height) {
                mapsWrapperElement.style.height = nextProps.height + "px";
                mapsElement.style.height = nextProps.height + "px";
            }
        }
    }

    render() {
        const { regular, actual, home, work } = this.props;
        const { actualUpdateTime, allPoints } = this.props;

        const points = allPoints.map(pairToPoint);
        const centerPoint = getCenter(points);
        const radius = getRadius(points, centerPoint);
        const zoom = getZoom(radius);

        if (allPoints.length === 0) {
            return (
                <div className="empty-content" style={{ height: window.innerHeight - BAR_HEIGHT * 3 }}>
                    User has no actual or regular coordinates
                </div>
            );
        } else {
            return (
                <div className="map-content" style={{ width: this.props.width + "px", float: "right" }}>
                    <Map
                        onAPIAvailable={function () {
                            console.info("Yandex Maps API loaded");
                        }}
                        height={window.innerHeight - 24 - 60 - 2 * BAR_HEIGHT}
                        width={this.props.width + "px"}
                        state={mapState}
                        center={pointToPair(centerPoint)}
                        zoom={zoom}
                    >
                        {regular.map((point, i) => {
                            return (
                                <Marker
                                    key={`regular_${i}`}
                                    lat={point.latitude}
                                    lon={point.longitude}
                                    onClick={() =>
                                        this.props.onRegularClick(
                                            point.update_time,
                                            point.latitude,
                                            point.longitude,
                                            "Regular"
                                        )
                                    }
                                >
                                    <MarkerLayout>{geoMarkerRegular}</MarkerLayout>
                                </Marker>
                            );
                        })}
                        {actual.map(([lat, lon], i) => {
                            return (
                                <Marker
                                    key={`actual_${i}`}
                                    lat={lat}
                                    lon={lon}
                                    onClick={() => this.props.onRegularClick(actualUpdateTime, lat, lon, "Actual")}
                                >
                                    <MarkerLayout>{geoMarkerActual}</MarkerLayout>
                                </Marker>
                            );
                        })}
                        {[home].filter(Boolean).map((eachLocation) => {
                            return (
                                <Marker
                                    key="home"
                                    lat={eachLocation.lat}
                                    lon={eachLocation.lon}
                                    onClick={() =>
                                        this.props.onRegularClick(undefined, eachLocation.lat, eachLocation.lon, "Home")
                                    }
                                >
                                    <MarkerLayout>{geoMarkerHome}</MarkerLayout>
                                </Marker>
                            );
                        })}
                        {[work].filter(Boolean).map((eachLocation) => {
                            return (
                                <Marker
                                    key="work"
                                    lat={eachLocation.lat}
                                    lon={eachLocation.lon}
                                    onClick={() =>
                                        this.props.onRegularClick(undefined, eachLocation.lat, eachLocation.lon, "Work")
                                    }
                                >
                                    <MarkerLayout>{geoMarkerWork}</MarkerLayout>
                                </Marker>
                            );
                        })}
                    </Map>
                    <GeoInfoBar value={this.props.pointInfo} uid={this.props.uid} />
                </div>
            );
        }
    }
}
