import React, { Component } from "react";
import _ from "lodash";
import { connect } from "react-redux";
import SearchPanel from "./search/SearchPanel";
import InfoPanel from "./info/InfoPanel";
import D3Graph from "./viewer/D3Graph";
import LoadingSpinner from "../common/LoadingSpinner";
import "./Graph.scss";
import { fetchGraph } from "./search/actions";
import { selectEdge, unselectIdAndEdge } from "./selectActions";
import Highcharts from "highcharts";
import HighchartsReact from "highcharts-react-official";
import FontAwesome from "react-fontawesome";

import { NEIGHBOURS_MATCHING_SCOPE, SOUP_CONFIG_MATCHING_TYPE } from "src/graph/search/settings/searchParams";

import {
    BySourceGraphStyler,
    ByDiffGraphStyler,
    ByCryptaIdGraphStyler,
    ByUsageScopeGraphStyler,
    ByIndeviceScopeGraphStyler,
    SoupConfigGraphStyler,
    SoupConfigIndeviceGraphStyler,
    DefaultGraphStyler,
    ByActivityGraphStyler,
} from "src/graph/viewer/style";

import { selectIdAndInfo } from "src/graph/selectActions";
import { getGraphParameters } from "src/selectors/graph";
import { toggleTimelineChart } from "src/graph/timelineActions";

const spinner = (
    <div>
        <LoadingSpinner size="l" /> Loading user graph...
    </div>
);

@connect(
    (state) => ({
        parameters: getGraphParameters(state),
        graph: state.graph,
        dot: state.graphExport,
        idsInfo: state.graphInfo.idsInfo,
        timeline: state.timeline,
    }),
    (dispatch) => ({
        onSelectId: (id) => dispatch(selectIdAndInfo(id)),
        onSelectEdge: (edge) => dispatch(selectEdge(edge)),
        onBackgroundClick: () => dispatch(unselectIdAndEdge()),
        fetchGraph: () => dispatch(fetchGraph()),
        clickTimelineButton: () => dispatch(toggleTimelineChart()),
    })
)
export default class Graph extends Component {
    componentDidUpdate(prevProps) {
        const { parameters, fetchGraph } = this.props;
        if (parameters !== prevProps.parameters) {
            fetchGraph();
        }
    }

    componentDidMount() {
        this.props.fetchGraph();
    }

    render() {
        let viewerArea;
        if (!_.isUndefined(this.props.graph.vertices)) {
            if (this.props.graph.isFetching) {
                viewerArea = Graph.renderLoadingInfo(spinner);
            } else {
                if (this.props.timeline.timelineChartToggled) {
                    viewerArea = this.renderActivityChart();
                } else {
                    viewerArea = this.renderGraph();
                }
            }
        }
        let timelineIcon = this.renderTimelineIcon();

        return (
            <div>
                <div className="left-panel">
                    <SearchPanel />
                    <div className="identifiers-panel">
                        <InfoPanel />
                    </div>
                </div>

                <div className="right-panel">
                    {viewerArea}
                    {timelineIcon}
                </div>
            </div>
        );
    }

    renderTimelineIcon() {
        let enabled = this.props.graph.isFetching ? { display: "none" } : {};
        return (
            <div className="timeline-button" style={enabled} onClick={() => this.props.clickTimelineButton()}>
                <FontAwesome name="history" />
            </div>
        );
    }

    renderActivityChart(viewerArea) {
        if (this.props.timeline.hasActivityData) {
            viewerArea = this.props.timeline.chartData.map((chart) => (
                <div key={chart.title}>
                    <HighchartsReact highcharts={Highcharts} options={chart} />
                </div>
            ));
        } else {
            viewerArea = <div className="loader">Activity data is not available</div>;
        }
        return viewerArea;
    }

    renderGraph() {
        let { idType, matchType, matchScope } = this.props.parameters;
        let styler;
        let stylersAvailable = [];
        if (this.props.parameters.showDiff) {
            styler = new ByDiffGraphStyler();
        } else {
            if (matchScope === NEIGHBOURS_MATCHING_SCOPE || idType === "merge_key") {
                styler = new ByCryptaIdGraphStyler();
            } else if (matchType === SOUP_CONFIG_MATCHING_TYPE) {
                styler = new SoupConfigGraphStyler();
                stylersAvailable.push(new SoupConfigIndeviceGraphStyler());
            } else {
                styler = new BySourceGraphStyler();
                stylersAvailable.push(new ByUsageScopeGraphStyler());
                stylersAvailable.push(new ByIndeviceScopeGraphStyler());
                stylersAvailable.push(new DefaultGraphStyler());
                stylersAvailable.push(new ByActivityGraphStyler());
                stylersAvailable.push(new ByCryptaIdGraphStyler());
            }
        }
        // at least one styler by default
        stylersAvailable.push(styler);

        if (this.props.parameters.showDiff === this.props.graph.isDiffGraph) {
            return (
                <D3Graph
                    vertices={this.props.graph.vertices}
                    edges={this.props.graph.edges}
                    isFetching={this.props.isFetching}
                    onSelectNode={this.props.onSelectId}
                    onSelectLink={this.props.onSelectEdge}
                    onBackgroundClick={this.props.onBackgroundClick}
                    graphStyler={styler}
                    stylersAvailable={stylersAvailable}
                    idsInfo={this.props.idsInfo}
                />
            );
        } else {
            return <div />;
        }
    }

    static renderLoadingInfo(info) {
        return <div className="loader">{info}</div>;
    }
}
