import _ from "lodash";

export function toVertex(idValue, idType) {
    return {
        idValue: idValue,
        idType: idType,
    };
}

export function vertexId(idValue, idType) {
    return idValue + "(" + idType.toUpperCase() + ")";
}

export function equals(id1, id2) {
    return _.isEqual(toVertex(id1.idValue, id1.idType), toVertex(id2.idValue, id2.idType));
}

export function getShortInfo(v, idsInfo) {
    let idInfo = idsInfo.find((info) => equals(info.id, v));
    if (idInfo === undefined) {
        return [];
    }

    let shortInfo = [];

    const tryAddInfo = (label, field) => {
        let info = _.get(idInfo.info, field);
        if (info) {
            shortInfo.push({
                label: label,
                info: info,
            });
        }
    };

    const { idType } = v;
    if (idType === "yandexuid") {
        tryAddInfo("Ua profile", "ua_profile");
        tryAddInfo("Main region", "main_region");
        tryAddInfo("Webview", "webview");
    } else if (idType === "idfa" || idType === "gaid") {
        tryAddInfo("Ua profile", "ua_profile");
        tryAddInfo("Main region", "main_region");
    } else if (idType === "uuid") {
        tryAddInfo("App", "app_id");
    }

    return shortInfo;
}

export function inlineGraphLabel(v, idsInfo) {
    let label = "";
    if (v.idType === "yandexuid" || v.idType === "idfa" || v.idType === "gaid" || v.idType === "uuid") {
        // possibly has info to display
        label = getLabel(v, idsInfo);
    } else if (v.idType === "mm_device_id") {
        // useless id to display
        label = v.idType.toUpperCase();
    } else {
        label = defaultInlineGraphLabel(v);
    }

    if (v.overlimitCount !== 0) {
        label += `[OVERLIMIT: ${v.overlimitCount}]`;
    }
    return label;
}

export function extractFullTextSearch(vertex, vertexInfo) {
    let overlimitFullTextToken = vertex.overlimitCount > 0 ? "overlimit" : "";

    return JSON.stringify(Object.values(vertex)) + JSON.stringify(Object.values(vertexInfo)) + overlimitFullTextToken;
}

function defaultInlineGraphLabel(v) {
    const idLength = v.idValue.length;
    let visibleIdValue = v.idValue;
    if (idLength > 30) {
        visibleIdValue = v.idValue.substr(0, 5) + "..." + v.idValue.substr(idLength - 5);
    }
    return visibleIdValue + "(" + v.idType.toUpperCase() + ")";
}

function shortIdType(idType) {
    if (idType === "uuid") {
        return "APP";
    } else if (idType === "yandexuid") {
        return "Y";
    } else {
        return idType.toUpperCase();
    }
}

function joinUaParts(...parts) {
    return parts
        .filter(function (el) {
            return el != null && el !== "";
        })
        .join("|");
}

function getBrowserInfo(yuidInfo) {
    let browser = _.get(yuidInfo, "browser") || "";
    if (browser !== "") {
        let browserVersion = _.get(yuidInfo, "browser_version") || "";
        let webview = _.get(yuidInfo, "webview") === true ? "webview" : "";

        return joinUaParts(browser, browserVersion, webview);
    } else {
        return "";
    }
}

function getLabel(v, idsInfo) {
    const { idType } = v;

    let idInfo = idsInfo.find((info) => equals(info.id, v));
    if (idInfo === undefined) {
        return shortIdType(idType);
    }

    if (idType === "yandexuid") {
        let uaProfile = _.get(idInfo.info, "ua_profile") || "";
        let browserInfo = getBrowserInfo(idInfo.info);
        let activityType = _.get(idInfo.info, "ip_activity_type") || "";
        if (activityType !== "") {
            activityType = "[" + activityType + "]";
        }
        return joinUaParts(uaProfile, browserInfo) + activityType;
    } else if (idType === "idfa" || idType === "gaid") {
        let uaProfile = _.get(idInfo.info, "ua_profile") || "";
        return uaProfile.toUpperCase();
    } else if (idType === "uuid") {
        let appId = _.get(idInfo.info, "app_id") || "";
        if (appId !== "") {
            return "APP(" + appId + ")";
        } else {
            return "";
        }
    } else {
        return "";
    }
}
