import React, { Component } from "react";

import { isEmpty } from "lodash";
import CopyDotButton from "./CopyDotButton";
import { clickSearchButton, clickSearchDiffButton } from "./actions";
import { connect } from "react-redux";

import "./SearchPanel.scss";
import "./ValueInput.scss";

import Panel from "@lib/components/panel";
import ValueInput from "./ValueInput";
import ActionButton from "../../common/ActionButton";
import { initGraphSettings } from "./settings/settingsActions";
import SelectInput from "./SelectInput";

import { getGraphParameters } from "selectors";
import { setInQuery } from "actions";
import { MD5 } from "object-hash";
import { GRAPH_V2_CUSTOM_PATH } from "src/common/Names";

@connect(
    (state) => ({
        parameters: getGraphParameters(state),
        showDiff: getGraphParameters(state).showDiff,
        dot: state.graphExport,
        matchTypes: state.graphSettings.matchTypes,
    }),
    (dispatch) => ({
        changeParameter: (key) => (value) => dispatch(setInQuery(key, value)),
        clickSearchButton: () => dispatch(clickSearchButton()),
        clickSearchDiffButton: () => dispatch(clickSearchDiffButton()),
        showDiffPanel: () => dispatch(setInQuery("showDiff", "true")),
        hideDiffPanel: () => dispatch(setInQuery("showDiff", "false")),
        initGraphSettings: () => dispatch(initGraphSettings()),
    })
)
class SearchPanel extends Component {
    componentDidMount() {
        this.props.initGraphSettings();
    }

    componentDidUpdate(prevProps, prevState) {
        // This is useful when there is some invalid uidType in query link
        // Invalid uidType replaced by the first of selectedIdTypes options
        // during firs loading of the page
        if (isEmpty(prevProps.matchTypes) && !isEmpty(this.props.matchTypes)) {
            this.updateDependentSelection(this.props.parameters["matchType"], "")
        }
    }

    updateMatchType(newMatchType, suffix) {
        this.props.changeParameter('matchType' + suffix)(newMatchType);
        this.updateDependentSelection(newMatchType, suffix)
    }

    updateDependentSelection(newMatchType, suffix) {
        const currentIdType = this.props.parameters["idType" + suffix];
        const currentMatchScope = this.props.parameters["matchScope" + suffix];

        let selectOptions = this.extractSelectOptions(newMatchType);

        let validIdType = selectOptions.idTypesOptions.includes(currentIdType)
            ? currentIdType
            : selectOptions.idTypesOptions[0];

        let validMatchScope = selectOptions.matchScopesOptions.includes(currentMatchScope)
            ? currentMatchScope
            : selectOptions.matchScopesOptions[0];

        this.props.changeParameter("uidType" + suffix)(validIdType);
        this.props.changeParameter("matchScope" + suffix)(validMatchScope);
    }

    render() {
        let { showDiff } = this.props;
        let widthClass = showDiff ? "expanded-search-panel" : "default-search-panel";

        return (
            <div className={widthClass}>
                <Panel>
                    <div className="show-diff-label">
                        {!showDiff ? (
                            <div onClick={() => this.props.showDiffPanel()}>Compare to another graph >></div>
                        ) : (
                            <div onClick={() => this.props.hideDiffPanel()}>{"<< Hide comparison"}</div>
                        )}
                    </div>
                    {this.renderInputGroup()}
                    {this.renderButtonGroup()}
                </Panel>
            </div>
        );
    }

    renderInputs(suffix) {
        const { parameters, changeParameter } = this.props;

        const matchType = parameters["matchType" + suffix];
        const selectOptions = this.extractSelectOptions(matchType);

        const validIdType = selectOptions.idTypesOptions.includes(parameters["idType"])
            ? parameters["idType"]
            : selectOptions.idTypesOptions[0];

        const showIdInput = matchType !== "soup_config";
        const showCustomPath = matchType === GRAPH_V2_CUSTOM_PATH;
        const showMatchScope = matchType !== "soup" && matchType !== "soup_config";
        const showSoupDepth = matchType === "soup" || matchType === "antifraud";

        // The MD5 hack is to hack inputs to re-render
        return (
            <div key={"inputs-" + suffix} className="input-group">
                <div style={{display: showIdInput ? "block" : "none"}}>
                    <ValueInput
                        key="id_value"
                        placeholder="[id value]"
                        value={parameters["idValue" + suffix]}
                        onChange={changeParameter("uid" + suffix)}
                    />
                </div>
                <div style={{display: showIdInput ? "block" : "none"}}>
                    <SelectInput
                        key={MD5(selectOptions.idTypesOptions)}
                        items={selectOptions.idTypesOptions}
                        value={isEmpty(suffix) ? validIdType : parameters["idType" + suffix]}
                        onChange={changeParameter("uidType" + suffix)}
                    />
                </div>
                <SelectInput
                    key={MD5(selectOptions.matchTypeOptions)}
                    items={selectOptions.matchTypeOptions}
                    value={matchType}
                    onChange={(value) => this.updateMatchType(value, suffix)}
                />
                <div style={{display: showCustomPath ? "block" : "none"}}>
                    <ValueInput
                        key="custom_path"
                        placeholder="[custom yt path]"
                        value={parameters["customMatchType" + suffix]}
                        onChange={changeParameter("customMatchType" + suffix)}
                    />
                </div>

                <div style={{display: showMatchScope ? "block" : "none"}}>
                    <SelectInput
                        key={MD5(selectOptions.matchScopesOptions)}
                        items={selectOptions.matchScopesOptions}
                        value={parameters["matchScope" + suffix]}
                        onChange={changeParameter("matchScope" + suffix)}
                    />
                </div>

                <div style={{display: showSoupDepth ? "block" : "none"}}>
                    <ValueInput
                        key="soup_depth"
                        placeholder="[graph depth]"
                        value={parameters["depth" + suffix]}
                        onChange={changeParameter("depth" + suffix)}
                    />
                </div>

            </div>
        );
    }

    renderInputGroup() {
        return (
            <div>
                {this.renderInputs("")}
                {this.props.showDiff && this.renderInputs("Other")}
            </div>
        );
    }

    extractSelectOptions(selectedMatchType) {
        let matchTypeOptions = this.props.matchTypes.map((mt) => mt.matchType);
        let selectedMatchTypeConfig = this.props.matchTypes.find((mt) => mt.matchType === selectedMatchType);
        return {
            matchTypeOptions: matchTypeOptions,
            idTypesOptions: selectedMatchTypeConfig !== undefined ? selectedMatchTypeConfig.idTypes : [],
            matchScopesOptions: selectedMatchTypeConfig !== undefined ? selectedMatchTypeConfig.matchScopes : [],
            otherInputsDisabled: selectedMatchTypeConfig === undefined,
        };
    }

    renderButtonGroup() {
        let onSearchButtonClick = this.props.showDiff ? this.props.clickSearchDiffButton : this.props.clickSearchButton;
        let searchButtonTitle = this.props.showDiff ? "Compare graphs" : "Get graph";

        let copyDotButton;
        if (this.props.showDiff) {
            copyDotButton = (
                <CopyDotButton elementId={"diff"} hint={".dot (diff)"} data={this.props.dot.graphsDiff.content} />
            );
        } else {
            copyDotButton = (
                <CopyDotButton elementId={"graph"} hint={".dot (graph)"} data={this.props.dot.graph.content} />
            );
        }

        return (
            <div className="button-group">
                <ActionButton
                    onClick={() => onSearchButtonClick()}
                    title={searchButtonTitle}
                    displayTitle={searchButtonTitle}
                />
                {copyDotButton}
            </div>
        );
    }
}

export default SearchPanel;
