import _ from "lodash";
import { exportGraph, exportGraphsDiff } from "../../../utils/apiClient";

export const REQUEST_GRAPH_EXPORT = "REQUEST_GRAPH_EXPORT";
function requestGraphExport(uid, uidType, matchType, matchScope) {
    return {
        type: REQUEST_GRAPH_EXPORT,
        uid,
        uidType,
        matchType,
        matchScope,
    };
}

export const RECEIVE_GRAPH_EXPORT = "RECEIVE_GRAPH_EXPORT";
function receiveGraphExport(dot) {
    return {
        type: RECEIVE_GRAPH_EXPORT,
        dot,
    };
}

export const INVALIDATE_GRAPH_EXPORT = "INVALIDATE_GRAPH_EXPORT";
export function invalidateGraphExport() {
    return {
        type: INVALIDATE_GRAPH_EXPORT,
    };
}

export const REQUEST_DIFF_EXPORT = "REQUEST_DIFF_EXPORT";
function requestDiffExport(idValue1, idType1, matchType1, matchScope1, idValue2, idType2, matchType2, matchScope2) {
    return {
        type: REQUEST_DIFF_EXPORT,
        idValue1,
        idType1,
        matchType1,
        matchScope1,
        idValue2,
        idType2,
        matchType2,
        matchScope2,
    };
}

export const RECEIVE_DIFF_EXPORT = "RECEIVE_DIFF_EXPORT";
function receiveDiffExport(dot) {
    return {
        type: RECEIVE_DIFF_EXPORT,
        dot,
    };
}

export const INVALIDATE_DIFF_EXPORT = "INVALIDATE_DIFF_EXPORT";
export function invalidateDiffExport() {
    return {
        type: INVALIDATE_DIFF_EXPORT,
    };
}

export function fetchGraphExport(uid, uidType, matchType, matchScope, depth) {
    return function (dispatch) {
        dispatch(requestGraphExport(uid, uidType, matchType, matchScope, depth));

        return exportGraph(uid, uidType, matchType, matchScope, depth)
            .then(
                // Do not use catch yet
                // https://github.com/facebook/react/issues/6895
                (response) => response,
                (error) => console.error("Error fetching graph .dot:", error)
            )
            .then((response) => {
                let dot = "";

                if (response) {
                    dot = response;
                }

                dispatch(receiveGraphExport(dot));
            });
    };
}

export function fetchDiffExport(
    idValue1,
    idType1,
    matchType1,
    matchScope1,
    depth1,
    idValue2,
    idType2,
    matchType2,
    matchScope2,
    depth2
) {
    return function (dispatch) {
        dispatch(
            requestDiffExport(
                idValue1,
                idType1,
                matchType1,
                matchScope1,
                depth1,
                idValue2,
                idType2,
                matchType2,
                matchScope2,
                depth2
            )
        );

        return exportGraphsDiff(
            idValue1,
            idType1,
            matchType1,
            matchScope1,
            depth1,
            idValue2,
            idType2,
            matchType2,
            matchScope2,
            depth2
        )
            .then(
                // Do not use catch yet
                // https://github.com/facebook/react/issues/6895
                (response) => response,
                (error) => console.error("Error fetching graphs diff .dot:", error)
            )
            .then((response) => {
                let dot = "";

                if (response) {
                    dot = response;
                }

                dispatch(receiveDiffExport(dot));
            });
    };
}

function shouldFetchExport(dot) {
    if (dot.isFetching) {
        return false;
    } else if (_.isEmpty(dot.content)) {
        return true;
    }

    return false;
}

export function fetchGraphExportIfNeeded(id, idType, matchingType, matchingScope, depth) {
    return function (dispatch, getState) {
        if (shouldFetchExport(getState().graphExport.graph)) {
            return dispatch(fetchGraphExport(id, idType, matchingType, matchingScope, depth));
        } else {
            return Promise.resolve();
        }
    };
}

export function fetchDiffExportIfNeeded(
    idValue1,
    idType1,
    matchType1,
    matchScope1,
    depth1,
    idValue2,
    idType2,
    matchType2,
    matchScope2,
    depth2
) {
    return function (dispatch, getState) {
        if (shouldFetchExport(getState().graphExport.graphsDiff)) {
            return dispatch(
                fetchDiffExport(
                    idValue1,
                    idType1,
                    matchType1,
                    matchScope1,
                    depth1,
                    idValue2,
                    idType2,
                    matchType2,
                    matchScope2,
                    depth2
                )
            );
        } else {
            return Promise.resolve();
        }
    };
}
