import { get as _get, isEmpty, isUndefined } from "lodash";
import { getGraphById, getGraphsDiff } from "../../../utils/apiClient";
import { groupBy } from "../../../utils/graphUtils";
import { toVertex, vertexId } from "../../info/vertexId";
import { resetCryptaIdSelection, selectCryptaId } from "../../selectActions";
import { createAction } from "redux-actions";
import { equals, extractFullTextSearch, inlineGraphLabel } from "src/graph/info/vertexId";
import { NEIGHBOURS_MATCHING_SCOPE } from "src/graph/search/settings/searchParams";
import { GRAPH_V2_CUSTOM_PATH } from "src/common/Names";
import { getIdTypeActivityCharts } from "src/graph/search/loader/activityChart";
import { addErrorNotification } from "../../../utils/notifications/notificationsState";

export const requestGraph = createAction("graph requested");
export const receiveGraph = createAction("graph received");
export const failGraph = createAction("graph failed");
export const invalidateGraph = createAction("graph invalidated");

export function fetchGraphById(parameters) {
    return (dispatch) => {
        dispatch(invalidateGraph({ isDiffGraph: false }));
        dispatch(resetCryptaIdSelection());
        dispatch(requestGraph({ isDiffGraph: false }));
        let { idValue, idType, matchType, matchScope, depth, customMatchType } = parameters;

        let realMatchType = matchType;
        if (matchType === GRAPH_V2_CUSTOM_PATH) {
            realMatchType = customMatchType;
        }

        if (idType === "merge_key") {
            matchScope = NEIGHBOURS_MATCHING_SCOPE;
        }

        return getGraphById(idValue, idType, realMatchType, matchScope, depth)
            .then((response) => {
                receiveGraphFromResponse(response.obj, false, matchScope, dispatch);
            })
            .catch((error) => {
                handleReceiveError(dispatch, error, idType, idValue);
            });
    };
}

export function fetchGraphsDiff(parameters) {
    return (dispatch) => {
        dispatch(invalidateGraph({ isDiffGraph: true }));
        dispatch(resetCryptaIdSelection());
        dispatch(requestGraph({ isDiffGraph: true }));

        let { idValue, idType, matchType, matchScope, depth, customMatchType } = parameters;
        let {
            idValueOther,
            idTypeOther,
            matchTypeOther,
            matchScopeOther,
            depthOther,
            customMatchTypeOther,
        } = parameters;

        let realMatchType = matchType;
        let realMatchTypeOther = matchTypeOther;

        if (matchType === GRAPH_V2_CUSTOM_PATH) {
            realMatchType = customMatchType;
        }

        if (matchTypeOther === GRAPH_V2_CUSTOM_PATH) {
            realMatchTypeOther = customMatchTypeOther;
        }

        return getGraphsDiff(
            idValue,
            idType,
            realMatchType,
            matchScope,
            depth,
            idValueOther,
            idTypeOther,
            realMatchTypeOther,
            matchScopeOther,
            depthOther
        )
            .then((response) => response)
            .then((response) => {
                receiveGraphFromResponse(response.obj, true, "", dispatch);
            })
            .catch((error) => {
                handleReceiveError(dispatch, error, idType, idValue);
            });
    };
}

function receiveGraphFromResponse(graph, isDiffGraph, matchingScope, dispatch) {
    let vertices = [];
    let edges = [];
    let idsInfo = graph.idsInfo;
    let componentsInfo = graph.componentsInfo;

    graph.graphComponents.forEach((component) => {
        let cryptaId = component.cryptaId;
        let componentVertices = component.vertices.map((vertex) => {
            return { ...vertex, cryptaId: cryptaId };
        });
        let componentEdges = component.edges.map((edge) => {
            return { ...edge, cryptaId: cryptaId };
        });

        vertices = vertices.concat(componentVertices);
        edges = edges.concat(componentEdges);

        let componentInfo = _get(componentsInfo, cryptaId) || {};
        componentsInfo[cryptaId] = componentInfo;

        componentInfo.idList = groupBy(componentVertices, "idType");
    });

    if (matchingScope === NEIGHBOURS_MATCHING_SCOPE) {
        let edgesBetween = graph.edgesBetweenComponents;
        edgesBetween.forEach((edge) => {
            let info = graph.mergeInfo.find((info) => info.mergeKey === edge.mergeKey);
            if (info) {
                edge.status = info.status;
                edge.strength = info.strength;
                edge.weightBetweenComponents = info.weight;
                edge.scoreGain = info.scoreGain;
                edge.similarityResults = info.similarityResults;
            }
        });
        edges = edges.concat(edgesBetween);
        edges.forEach((edge) => {
            // sometimes edge comes without corresponding vertices for some reason
            let v1 = vertices.find((v) => v.idValue === edge.id1 && v.idType === edge.id1Type);
            if (v1 === undefined) {
                vertices.push(toVertex(edge.id1, edge.id1Type));
            }
            let v2 = vertices.find((v) => v.idValue === edge.id2 && v.idType === edge.id2Type);
            if (v2 === undefined) {
                vertices.push(toVertex(edge.id2, edge.id2Type));
            }
        });
    }
    let idTypeActivityCharts = getIdTypeActivityCharts(edges);
    let hasActivityData = idTypeActivityCharts.some((data) => !(data.xAxis.categories.length === 0));

    edges = edges.map((edge) => {
        return {
            ...edge,
            source: vertexId(edge.id1, edge.id1Type),
            target: vertexId(edge.id2, edge.id2Type),
            fullTextSearch: JSON.stringify(edge),
        };
    });

    let vertexDegrees = {};

    function updateDegree(v) {
        if (v in vertexDegrees) {
            vertexDegrees[v] = vertexDegrees[v] + 1;
        } else {
            vertexDegrees[v] = 1;
        }
    }

    for (const edge of edges) {
        updateDegree(edge.source);
        updateDegree(edge.target);
    }

    vertices = vertices.map((vertex) => {
        let fullId = vertexId(vertex.idValue, vertex.idType);

        let vertexInfo = idsInfo.find((info) => equals(info.id, vertex)) || {};
        let vertexDegree = vertexDegrees[fullId];

        return {
            ...vertex,
            fullId: fullId,
            degree: vertexDegree === undefined ? 1 : vertexDegree, // single vertex case
            inlineGraphLabel: inlineGraphLabel(vertex, idsInfo),
            fullTextSearch: extractFullTextSearch(vertex, vertexInfo),
        };
    });

    let firstCryptaId = Object.keys(componentsInfo)[0];

    dispatch(
        receiveGraph({
            vertices: vertices,
            edges: edges,
            idsInfo: idsInfo,
            componentsInfo: componentsInfo,
            isDiffGraph: isDiffGraph,
            hasActivityData: hasActivityData,
            activityChartData: idTypeActivityCharts,
        })
    );
    dispatch(selectCryptaId(firstCryptaId));
}

function handleReceiveError(dispatch, error, uidType, uid) {
    let status = error.status;

    if (!isUndefined(status) && status !== 404) {
        dispatch(addErrorNotification(error, `Failed to fetch graph: ${uidType}=${uid}`));
    }

    dispatch(failGraph());
}

function shouldFetchGraph(state) {
    const graph = state.graph;

    if (graph.isFetching) {
        return false;
    } else if (isUndefined(graph.graphComponents) || isEmpty(graph.graphComponents)) {
        return true;
    }

    return false;
}

export const fetchGraphsDiffIfNeeded = (parameters) => (dispatch, getState) => {
    if (shouldFetchGraph(getState())) {
        return dispatch(fetchGraphsDiff(parameters));
    } else {
        return Promise.resolve();
    }
};

export const fetchGraphByIdIfNeeded = (parameters) => (dispatch, getState) => {
    if (shouldFetchGraph(getState())) {
        return dispatch(fetchGraphById(parameters));
    } else {
        return Promise.resolve();
    }
};
