import React, { Component } from "react";
import * as d3 from "d3";
import _ from "lodash";

import "./D3Graph.scss";
import { vertexId } from "src/graph/info/vertexId";
import {edgeName} from "../info/edgeId";

class D3Graph extends Component {
    constructor(props) {
        super(props);
        this.state = {
            graphStyler: this.props.graphStyler,
            searchText: "",
        };
        this.updateStyler = this.updateStyler.bind(this);
        this.searchInGraph = this.searchInGraph.bind(this);
    }

    updateStyler(event) {
        this.setState({
            graphStyler: this.props.stylersAvailable.find((s) => s.name === event.target.value),
        });
    }

    shouldComponentUpdate(nextProps, nextState) {
        // search must not rerender the whole graph
        // it will be colored using d3 state instead of react state
        return nextState.searchText === this.state.searchText;
    }

    componentDidMount() {
        this.renderGraph(this.props.vertices, this.props.edges);
    }

    componentDidUpdate() {
        this.renderGraph(this.props.vertices, this.props.edges);
    }

    renderGraph(_vertices, _edges) {
        let svg = d3.select("#d3-graph");
        let width = +svg.attr("width");
        let height = +svg.attr("height");

        svg.selectAll("*").remove();

        let vertices = _.map(_vertices, (each) => {
            if (each.embedding) {
                return {
                    ...each,
                    x: (width * (each.embedding.value[0] + 5.0)) / 10.0,
                    y: (height * (each.embedding.value[1] + 5.0)) / 10.0,
                    vx: 0,
                    vy: 0,
                };
            } else {
                return {
                    ...each,
                };
            }
        });
        // making defensive copy because d3 modifies original data (source and target)
        // https://github.com/d3/d3-force#link_links
        let edges = _.map(_edges, (each) => {
            return { ...each };
        });

        let onSelectNode = this.props.onSelectNode;
        let onSelectLink = this.props.onSelectLink;
        let graphStyler = this.state.graphStyler;
        let onBackgroundClick = this.props.onBackgroundClick;
        let idsInfo = this.props.idsInfo;
        let self = this;

        if (!_.isEmpty(vertices) || !_.isEmpty(edges)) {
            this.prepareEdgeCurves(edges);

            let g = svg.append("g");

            let link = g
                .append("g")
                .attr("class", "links")
                .selectAll("g")
                .data(edges)
                .enter()
                .append("g")
                .attr("class", "userpath");
            let path = link
                .append("path")
                .attr("id", (d) => `${d.source}__to__${d.target}__${d.sourceType}__${d.logSource}`);

            let text = link.append("text").attr("dy", "-5");

            text.append("textPath")
                .attr("xlink:href", (d) => `#${d.source}__to__${d.target}__${d.sourceType}__${d.logSource}`)
                .attr("startOffset", "50%")
                .style("text-anchor", "middle")
                .style("line-height", "1.5")
                .text((d) => edgeName(d));

            graphStyler.defaultEdge(link);

            link.on("mouseover", function () {
                graphStyler.hoverEdge(d3.select(this));
            }).on("mouseout", function () {
                graphStyler.unhoverEdge(d3.select(this));
                if (self.state.searchText !== "") {
                    self.highlightFoundEdges(d3.select(this), self.state.searchText);
                }
            });

            link.on("click", (d) => {
                onSelectLink(d);
                d3.event.stopPropagation();
            });

            let node = g
                .append("g")
                .attr("class", "nodes")
                .selectAll("g")
                .data(vertices)
                .enter()
                .append("g")
                .call(d3.drag().on("start", dragstarted).on("drag", dragged).on("end", dragended));

            node.append("circle");
            node.append("text");
            graphStyler.defaultVertex(node);

            node.on("click", (d) => {
                onSelectNode(d);
                d3.event.stopPropagation();
            });

            node.on("mouseover", function () {
                graphStyler.hoverVertex(d3.select(this));
            }).on("mouseout", function () {
                graphStyler.unhoverVertex(d3.select(this));
                if (self.state.searchText !== "") {
                    self.highlightFoundVertices(d3.select(this), self.state.searchText);
                }
            });

            let {
                verticesStrength,
                verticesMaxDistance,
                edgesDistance,
                edgesStrength,
                linkArcCoefficient,
            } = graphStyler.layoutParams();

            let simulation = d3
                .forceSimulation(vertices)
                .force("charge", d3.forceManyBody().strength(verticesStrength).distanceMax(verticesMaxDistance))
                .force(
                    "link",
                    d3
                        .forceLink(edges)
                        .id((d) => vertexId(d.idValue, d.idType))
                        .strength(edgesStrength)
                        .distance(edgesDistance)
                        .iterations(6)
                )
                .force("center", d3.forceCenter(width / 2, 400))
                .on("tick", ticked);

            function linkArc(d) {
                // Rework arc algorithm; I suspect it is not optimal
                let dx = d.target.x - d.source.x,
                    dy = d.target.y - d.source.y,
                    dr = Math.sqrt(dx * dx + dy * dy),
                    arc = (dr * d.maxSameHalf) / (d.sameIndexCorrected * linkArcCoefficient);

                if (d.sameMiddleLink) {
                    arc = 0;
                }

                return (
                    "M" +
                    d.source.x +
                    "," +
                    d.source.y +
                    "A" +
                    arc +
                    "," +
                    arc +
                    " 0 0," +
                    d.sameArcDirection +
                    " " +
                    d.target.x +
                    "," +
                    d.target.y
                );
            }

            function ticked() {
                path.attr("d", linkArc);

                node.attr("transform", function (d) {
                    return "translate(" + d.x + "," + d.y + ")";
                });
            }

            function dragstarted(d) {
                if (!d3.event.active) simulation.alphaTarget(0.3).restart();
                d.fx = d.x;
                d.fy = d.y;
            }

            function dragged(d) {
                d.fx = d3.event.x;
                d.fy = d3.event.y;
            }

            function dragended(d) {
                simulation.stop();
            }

            svg.call(d3.zoom().on("zoom", zoom_actions));

            svg.on("click", () => {
                onBackgroundClick();
            });

            function zoom_actions() {
                g.attr("transform", d3.event.transform);
            }
        }
    }

    prepareEdgeCurves(edges) {
        if (_.isEmpty(edges)) {
            return;
        }

        _.each(edges, (link) => {
            // find other links with same target+source or source+target
            let same = _.filter(edges, {
                source: link.source,
                target: link.target,
            });
            let sameAlt = _.filter(edges, {
                source: link.target,
                target: link.source,
            });
            let sameAll = same.concat(sameAlt);

            _.each(sameAll, function (s, i) {
                let isReversed = s.source == link.source ? 0 : 1;
                s.sameIndex = i + 1;
                s.sameTotal = sameAll.length;
                s.sameTotalHalf = Math.ceil(s.sameTotal / 2);
                s.sameUneven = s.sameTotal % 2 !== 0;
                s.sameMiddleLink = s.sameUneven === true && s.sameTotalHalf === s.sameIndex;
                s.sameLowerHalf = s.sameIndex <= s.sameTotalHalf;
                s.sameArcDirection = s.sameLowerHalf ? isReversed : isReversed ^ 1;
                s.sameIndexCorrected = s.sameLowerHalf ? s.sameIndex : s.sameIndex - s.sameTotalHalf;
            });
        });

        let maxSame = _.sortBy(edges, (item) => item.sameTotal).reverse()[0].sameTotal;

        _.each(edges, function (link) {
            link.maxSameHalf = Math.floor(maxSame / 3);
        });
    }

    highlightFoundVertices(vertexNodes, textFilter) {
        let graphStyler = this.state.graphStyler;

        if (textFilter !== undefined && textFilter !== "") {
            vertexNodes
                .filter((data) => data.fullTextSearch && !data.fullTextSearch.includes(textFilter))
                .call((x) => graphStyler.unhighlightVertex(x));
            vertexNodes
                .filter((data) => data.fullTextSearch && data.fullTextSearch.includes(textFilter))
                .call((x) => graphStyler.highlightVertex(x));
        } else {
            // return to defaults
            vertexNodes.call((x) => graphStyler.defaultVertex(x));
        }
    }

    highlightFoundEdges(edgeNodes, textFilter) {
        let graphStyler = this.state.graphStyler;

        if (textFilter !== undefined && textFilter !== "") {
            edgeNodes
                .filter((data) => data.fullTextSearch && !data.fullTextSearch.includes(textFilter))
                .call((x) => graphStyler.unhighlightEdge(x));
            edgeNodes
                .filter((data) => data.fullTextSearch && data.fullTextSearch.includes(textFilter))
                .call((x) => graphStyler.highlightEdge(x));
        } else {
            // return to defaults
            edgeNodes.call((x) => graphStyler.defaultEdge(x));
        }
    }

    searchInGraph(textFilter) {
        this.setState({
            searchText: textFilter,
        });

        let svg = d3.select("#d3-graph");
        let vertexNodes = svg.select(".nodes").selectAll("g");
        let edgeNodes = svg.select(".links").selectAll("g");

        this.highlightFoundVertices(vertexNodes, textFilter);
        this.highlightFoundEdges(edgeNodes, textFilter);
    }

    render() {
        const emptyGraphCondition =
            _.isEmpty(this.props.vertices) && _.isEmpty(this.props.edges) && !this.props.isFetching;
        const stylers = this.props.stylersAvailable.map((st) => <option key={st.name}>{st.name}</option>);

        return (
            <div className="d3-graph-container">
                <input
                    placeholder="Search in graph"
                    className="in-graph-search-field"
                    onChange={(event) => this.searchInGraph(event.target.value)}
                />

                <select
                    className="in-graph-select-style-field"
                    value={this.state.graphStyler.name}
                    onChange={this.updateStyler}
                >
                    {stylers}
                </select>

                {emptyGraphCondition ? <div className="loader">User's graph is empty</div> : <div />}
                <svg id="d3-graph" width="900" height="1200" />
            </div>
        );
    }
}

export default D3Graph;
