import React, { useState } from "react";
import { connect } from "react-redux";
import { sortBy, isEmpty } from "lodash";
import InterestItem from "./InterestItem";
import PredictedLabel from "./PredictedLabel";
import { getEnvTypeState } from "../utils/selectors";

import "./Interests.scss";

const getLabUrl = (envType, segmentId) => {
    const labSuffix = envType === "production" ? "" : "-test";

    return `https://lab${labSuffix}.crypta.yandex-team.ru/segments?segment=${segmentId}`;
};

const Interests = connect((state) => ({
    envType: getEnvTypeState(state),
}))((props) => {
    const { profile, weighted, chronological, itemsToShow, showMoreLabel, showLessLabel, envType } = props;

    const [extraHidden, setExtraHidden] = useState(true);

    const extraItemClassSuffix = extraHidden ? "-hidden" : "";

    const items = weighted
        ? sortBy(profile.value, (item) => item.weight).reverse()
        : chronological
        ? sortBy(profile.value, (item) => item.timestamp).reverse()
        : sortBy(profile.value, (item) => item.name);

    const visibleItems = items.slice(0, itemsToShow);
    const extraItems = items.slice(itemsToShow);

    return (
        profile && (
            <div>
                {profile.timestamp && <PredictedLabel time={profile.timestamp} />}

                <div className="Interests-content">
                    <table className="Interests-table">
                        <tbody>
                            {visibleItems.map((item, index) => {
                                return (
                                    <tr key={index}>
                                        <td>
                                            <InterestItem
                                                item={item}
                                                weighted={weighted}
                                                labUrl={getLabUrl(envType, item.segmentId)}
                                            />
                                        </td>
                                    </tr>
                                );
                            })}

                            {extraItems.map((item, index) => {
                                return (
                                    <tr key={index} className={`interest-extra-item${extraItemClassSuffix}`}>
                                        <td>
                                            <InterestItem item={item} labUrl={getLabUrl(envType, item.segmentId)} />
                                        </td>
                                    </tr>
                                );
                            })}

                            {!isEmpty(extraItems) && (
                                <tr>
                                    <td onClick={() => setExtraHidden(!extraHidden)}>
                                        <div className="show-more">{extraHidden ? showMoreLabel : showLessLabel}</div>
                                    </td>
                                </tr>
                            )}
                        </tbody>
                    </table>
                </div>
            </div>
        )
    );
});

Interests.defaultProps = {
    itemsToShow: 100,
    showMoreLabel: "Show more",
    showLessLabel: "Show less",
};

export default Interests;
