import { createAction } from "redux-actions";
import {
    getExperiments,
    getExperimentsDescription,
    getMeProfile,
    getTransactions,
} from "src/utils/apiClient";
import { addErrorNotification, dropAllNotifications } from "../utils/notifications/notificationsState";

const requestMe = createAction("bb profile requested");
const receiveMe = createAction("bb profile received");

const requestExperiments = createAction("experiments requested");
const receiveExperiments = createAction("experiments received");

const requestExperimentsDescription = createAction("experiments description requested");
const receiveExperimentsDescription = createAction("experiments description received");

const requestTransactions = createAction("transactions requested");
const receiveTransactions = createAction("transactions received");

export function fetchMeProfile(uidType, uidValue) {
    return function (dispatch) {
        dispatch(dropAllNotifications());
        dispatch(requestMe());

        getMeProfile(uidType, uidValue)
            .then((response) => {
                let profile = {};

                if (response) {
                    profile = response;
                }

                dispatch(receiveMe(profile));
            })
            .catch((error) => {
                console.error("Error loading user main profile:", error);

                dispatch(receiveMe({}));

                if (error.status && error.status !== 404) {
                    dispatch(addErrorNotification(error, "Error loading profile"));
                }
            });
    };
}

export function fetchExperiments(uid) {
    return function (dispatch) {
        dispatch(requestExperiments());

        getExperiments(uid)
            .then((response) => {
                let payload = {};

                if (response) {
                    payload = response.obj;
                }

                let haystack = payload.map((item) => item.key);

                dispatch(receiveExperiments([payload, haystack, response.status]));
            })
            .catch((error) => {
                console.error("Error fetching experiments for uid", uid, error);

                if (error.status && ![403, 404].includes(error.status)) {
                    dispatch(addErrorNotification(error, "Failed to fetch experiments"));
                }
                dispatch(receiveExperiments([{}, error.status]));
            });
    };
}

export function fetchTransactions(uid) {
    return function (dispatch) {
        dispatch(requestTransactions());

        getTransactions(uid)
            .then((response) => {
                dispatch(receiveTransactions(response.obj));
            })
            .catch((error) => {
                dispatch(receiveTransactions([]));
            });
    }
}

export function fetchExperimentsDescription(uid) {
    return function (dispatch) {
        dispatch(requestExperimentsDescription());

        getExperimentsDescription(uid)
            .then((response) => {
                let payload = [];

                if (response) {
                    payload = response.obj;
                }

                dispatch(receiveExperimentsDescription(payload));
            })
            .catch((error) => {
                if (error.status && ![403, 404].includes(error.status)) {
                    dispatch(addErrorNotification(error, "Failed to fetch active experiments: uid=" + uid));
                }

                dispatch(receiveExperimentsDescription([]));
            });
    };
}

export {
    requestMe,
    receiveMe,
    requestExperiments,
    receiveExperiments,
    requestExperimentsDescription,
    receiveExperimentsDescription,
    requestTransactions,
    receiveTransactions,
};
