import React, { useEffect, useState } from "react";
import { connect, useDispatch } from "react-redux";
import { useTranslation } from "react-i18next";
import { isEmpty } from "lodash";
import { RadioButton, TextInput } from "@lib/components/lego";
import { fetchExperiments } from "src/me/actions";
import ExperimentsResult from "src/me/experiments/ExperimentsResult";
import { searchSubstring } from "src/me/experiments/searchSubstring";
import ExperimentsDescription from "src/me/experiments/ExperimentsDescription";

import "./Experiments.scss";

const Experiments = connect((state) => ({
    experiments: state.experiments,
}))((props) => {
    const { uid, cryptaId, experiments } = props;

    const [searchValue, setSearchValue] = useState("");
    const [searchResult, setSearchResult] = useState([]);
    const [activeUid, setActiveUid] = useState("uid");

    const dispatch = useDispatch();
    const { t } = useTranslation("profile");

    const handleEscapePressed = (event) => {
        if (event.keyCode === 27) {
            setSearchValue("");
        }
    };

    useEffect(() => {
        dispatch(fetchExperiments(uid));
    }, [uid]);

    useEffect(() => {
        document.addEventListener("keydown", handleEscapePressed, false);
        return () => {
            document.removeEventListener("keydown", handleEscapePressed, false);
        };
    }, []);

    useEffect(() => {
        let currentUid = activeUid === "cryptaId" ? cryptaId : uid;
        dispatch(fetchExperiments(currentUid));
    }, [activeUid]);

    const handleChange = (value) => {
        let result = searchSubstring(value, experiments.haystack);
        setSearchValue(value);
        setSearchResult(result);
    };

    const selectUidType = (value) => {
        setActiveUid(value);
    };

    let experimentParameters = {};
    if (!isEmpty(experiments.values)) {
        let exp_dict = {};

        experiments.values.forEach((item) => {
            exp_dict[item.key] = item.value;
        });

        searchResult.forEach((item) => {
            experimentParameters[item] = exp_dict[item];
        });
    }

    const placeholder = t("experimentsPlaceholder") + activeUid + " " + (activeUid === "cryptaId" ? cryptaId : uid);

    return (
        experiments.status !== 403 && (
            <>
                <div className="experiments-bar">
                    <div className="experiments-select-uid-type">
                        <RadioButton
                            value={activeUid}
                            size="s"
                            view="default"
                            className="select-sorting"
                            onChange={(event) => selectUidType(event.target.value)}
                            options={[
                                { value: "uid", children: t("by") + " yandexuid" },
                                { value: "cryptaId", children: t("by") + " CryptaID" },
                            ]}
                        />
                    </div>
                    <div className="experiments-search-box">
                        <TextInput
                            view="default"
                            size="m"
                            placeholder={experiments.loading ? "loading..." : placeholder}
                            value={searchValue}
                            onChange={(event) => handleChange(event.target.value)}
                            onClearClick={() => handleChange("")}
                            hasClear
                        />
                        <ExperimentsResult parameters={experimentParameters} target={searchValue} />
                    </div>
                </div>

                <ExperimentsDescription uid={activeUid === "cryptaId" ? cryptaId : uid} />
            </>
        )
    );
});

export default Experiments;
