import React, { useEffect } from "react";
import { connect, useDispatch } from "react-redux";
import { isEmpty, isUndefined } from "lodash";
import moment from 'moment';
import "moment/locale/ru";
import ProbInterests from "./ProbInterests";
import NewHeuristicInterests from "./NewHeuristicInterests";
import Dmp from "./Dmp";
import Loading from "@lib/components/loading";

import GenderIcon from "me/icon/gender";
import AgeIcon from "me/icon/age";
import IncomeIcon from "me/icon/income";

import PredictedLabel from "me/PredictedLabel";

import { COMMON_TO_BIGB_API_ID_MAP } from "src/common/Names";
import { fetchMeProfile } from "src/me/actions";
import Tile from "src/me/layout/Tile";
import { getUid } from "src/selectors";
import { getUidType } from "src/selectors";
import Experiments from "src/me/experiments/Experiments";
import Transactions from "src/me/transactions";

import "./style.scss";
import { useTranslation } from "react-i18next";
import TimedInterests from "./TimedInterests";
import Interests from "./Interests";

function ExperimentsRow(props) {
    const { uid, cryptaId } = props;

    const { t } = useTranslation("profile");

    return (
        <div className="grid-row">
            <div className="grid-col">
                <Tile title={t("experiments")}>
                    <Experiments uid={uid} cryptaId={cryptaId} />
                </Tile>
            </div>
        </div>
    );
}

function TransactionsRow(props) {
    const { uid } = props;

    return (
        <div className="grid-row">
            <div className="grid-col">
                <Transactions uid={uid} />
            </div>
        </div>
    );
}

function SocdemRow(props) {
    const { profile } = props;
    const { gender, age, income, exactDemographics } = profile;

    const { t } = useTranslation("profile");

    return (
        <div className="grid-row">
            <div className="grid-col">
                <Tile title={t("gender")}>
                    <PredictedLabel time={gender.timestamp} />
                    <GenderIcon values={gender} exact={exactDemographics.gender} />
                </Tile>
            </div>
            <div className="grid-col">
                <Tile title={t("age")}>
                    <PredictedLabel time={age.timestamp} />
                    <AgeIcon values={age} exact={exactDemographics.age6} />
                </Tile>
            </div>
            <div className="grid-col">
                <Tile title={t("income")}>
                    <PredictedLabel time={income.timestamp} />
                    <IncomeIcon values={income} exact={exactDemographics.income5} />
                </Tile>
            </div>
        </div>
    );
}

function SegmentsRow(props) {
    const { profile } = props;

    const { t } = useTranslation("profile");

    return (
        <React.Fragment>
            <div key="row-1" className="grid-row">
                <div key="heuristic" className="grid-col">
                    <Tile title={t("heuristic")}>
                        {profile.heuristic && profile.heuristic.value.length > 0 && (
                            <Interests profile={profile.heuristic} />
                        )}
                    </Tile>
                </div>
                <div key="new-heuristic" className="grid-col">
                    <Tile title={t("newHeuristic")}>
                        <NewHeuristicInterests
                            profile={profile.newHeuristic.common}
                            timestamp={profile.newHeuristic.timestamp}
                        />
                    </Tile>
                </div>
                <div key="orgs" className="grid-col">
                    <Tile title={t("organizations")}>
                        <NewHeuristicInterests
                            profile={profile.newHeuristic.organizations}
                            timestamp={profile.newHeuristic.timestamp}
                        />
                    </Tile>
                </div>
                <div key="probabilistic" className="grid-col">
                    <Tile title={t("probabilistic")}>
                        <ProbInterests profile={profile.probabilistic} />
                    </Tile>
                </div>
            </div>
            <div key="row-2" className="grid-row">
                <div key="short-term" className="grid-col">
                    <Tile title={t("shortTerm")}>
                        <TimedInterests profile={profile.shortterm} chronological />
                    </Tile>
                </div>
                <div key="long-term" className="grid-col">
                    <Tile title={t("longTerm")}>
                        <TimedInterests profile={profile.longterm} />
                    </Tile>
                </div>
                <div key="dmp" className="grid-col">
                    <Tile title={t("dmp")}>
                        {
                            // TODO indicate if there's no dmp
                        }
                        <Dmp profile={profile.dmp.value} />
                    </Tile>
                </div>
            </div>
        </React.Fragment>
    );
}

const Me = connect((state) => ({
    uid: getUid(state),
    uidType: getUidType(state),
    me: state.me,
}))((props) => {
    const { uid, uidType } = props;
    const { profile, loading } = props.me;

    const { t } = useTranslation("profile");

    const dispatch = useDispatch();

    const cryptaId = isUndefined(profile.indeviceIdentifiers) ? "" : profile.indeviceIdentifiers.cryptaId;

    useEffect(() => {
        dispatch(fetchMeProfile(COMMON_TO_BIGB_API_ID_MAP[uidType], uid));
    }, [uid]);
    useEffect(() => {
        moment.locale("en");
    }, []);

    return (
        <Loading loaded={!loading} size="m">
            { !loading &&
            <div className="me-container FadeIn">
                <ExperimentsRow uid={uid} cryptaId={cryptaId} />
                {!isEmpty(profile) ? (
                    <React.Fragment>
                        <SocdemRow profile={profile} />
                        <SegmentsRow profile={profile} />
                    </React.Fragment>
                ) : (
                    <div className="Me-not-loaded">{t("notLoaded")}</div>
                )}
                <TransactionsRow uid={uid}/>
            </div>
            }
        </Loading>
    );
});

export default Me;
