import React, { useCallback, useEffect, useState } from "react";
import Viva from "vivagraphjs";

import "./Graph.scss";

import skeleton from "./skeleton.png";

export function Graph({ nodes, edges, edgeLength = 200 }) {
    const [renderer, setRenderer] = useState(null);

    useEffect(() => {
        const container = document.querySelector(".Graph-Container");
        if (!container) {
            return null;
        }

        const graph = Viva.Graph.graph();
        const graphics = configureGraphics();

        nodes.forEach(node => graph.addNode(node.id, node));
        edges.forEach(edge => graph.addLink(edge.fromId, edge.toId, edge.data));

        const layout = Viva.Graph.Layout.forceDirected(graph, {
            gravity: -30,
            springCoeff: 0.0001,
            thetaCoeff: 0.8,
            springTransform: function (link, spring) {
                spring.length = link.data?.edgeLength ?? edgeLength;
            },
        });

        const renderer = Viva.Graph.View.renderer(graph, {
            container,
            graphics,
            layout,
            interactive: 'node drag'
        });

        precomputeLayout(layout, 1000, () => runRenderer(renderer));

        setRenderer(renderer);

        return () => {
            setRenderer(null);
            renderer.dispose();
        }
    }, [nodes, edges]);

    return (
        <div className="Graph">
            <div className="Graph-Container"/>
            { renderer && <GraphControls renderer={renderer} /> }
        </div>
    );
}

export function GraphControls({ renderer }) {
    const zoomIn = useCallback(() => {
        renderer.zoomIn();
    }, [renderer]);

    const zoomOut = useCallback(() => {
        renderer.zoomOut();
    }, [renderer]);

    return (
        <div className="Graph-Controls">
            <div
                className="Graph-ControlZoomIn"
                onClick={zoomIn}
            >
                +
            </div>
            <div
                className="Graph-ControlZoomOut"
                onClick={zoomOut}
            >
                -
            </div>
        </div>
    )
}

export function GraphSkeleton() {
    return (
        <img className="Graph" alt="" src={skeleton}/>
    );
}

function configureGraphics() {
    const graphics = Viva.Graph.View.svgGraphics();

    graphics
        .node(function (node) {
            const container = Viva.Graph.svg("g")
                .attr("data-id", node.data.id);

            container.classList.add("Graph-Node");

            if (node.data.imageHref) {
                // <image xlink:href="..." height="26" width="26" x="10" y="10"/>
                const image = Viva.Graph.svg("image")
                    .attr("height", node.data.imageSize)
                    .attr("width", node.data.imageSize)
                    .attr("x", -node.data.imageSize / 2)
                    .attr("y", -node.data.imageSize / 2)
                    .link(node.data.imageHref);

                container.appendChild(image);
            }

            if (node.data.text) {
                // <text x="20" y="35" class="small"></text>
                const text = Viva.Graph.svg("text")
                    .attr("text-anchor", "middle")
                    .attr("y", node.data.imageSize / 2 + 25); // radius + margin

                text.innerHTML = node.data.text;

                container.appendChild(text);
            }

            if (node.data.popup) {
                const popup = Viva.Graph.svg("foreignObject")
                    .attr("x", 0)
                    .attr("y", node.data.imageSize / 2 + 2)
                    .attr("height", 1)
                    .attr("width", 1);

                popup.classList.add("Graph-PopupWrapper");

                popup.innerHTML = `
                    <div class="Graph-Popup">
                        <div class="Graph-PopupHeader">${node.data.popup.header}</div>
                        <div class="Graph-PopupText">${node.data.popup.text}</div>
                    </div>
                `;

                container.appendChild(popup);
            }

            return container;
        })
        .placeNode((nodeUI, pos) => {
            nodeUI.attr("transform", `translate(${pos.x}, ${pos.y})`);
        })
        .link((link) => {
            return Viva.Graph.svg("line")
                .attr("stroke", link.data?.color ?? "#BCC0CB")
                .attr("stroke-width", link.data?.width ?? 1)
                .attr("stroke-dasharray", "2 1");
        });


    return graphics;
}

function precomputeLayout(layout, iterations, callback) {
    let i = 0;
    while (iterations > 0 && i < 10) {
        layout.step();
        iterations--;
        i++;
    }

    if (iterations > 0) {
        setTimeout(function () {
            precomputeLayout(layout, iterations, callback);
        }, 0);
    } else {
        callback();
    }
}

function runRenderer(renderer) {
    renderer.run();

    // Final bit: most likely graph will take more space than available
    // screen. Let's zoom out to fit it into the view:
    const graphRect = renderer.getLayout().getGraphRect();
    const graphSize = Math.min(graphRect.x2 - graphRect.x1, graphRect.y2 - graphRect.y1);
    const container = document.querySelector(".Graph-Container");
    const screenSize = Math.max(container.clientWidth, container.clientHeight);

    const desiredScale = screenSize / graphSize;
    zoomOut(desiredScale, 1);

    function zoomOut(desiredScale, currentScale) {
        // zoom API in vivagraph is silly. There is no way to pass transform
        // directly. Maybe it will be fixed in future, for now this is the best I could do:
        if (desiredScale < currentScale) {
            currentScale = renderer.zoomOut();
            setTimeout(function () {
                zoomOut(desiredScale, currentScale);
            }, 16);
        }
    }
}
